<?php
/**
 * Migration scripts for ColorMag theme.
 *
 * @package    ThemeGrill
 * @subpackage ColorMag
 * @since      ColorMag 3.0.0
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'ColorMag_Migration' ) ) {
	class ColorMag_Migration {

		public function __construct() {
			if ( self::maybe_run_migration() || self::colormag_demo_import_migration() ) {
				add_action( 'after_setup_theme', array( $this, 'colormag_major_update_v1_customize_migrate' ) );
				add_action( 'after_setup_theme', array( $this, 'colormag_social_icons_control_migrate' ) );
				add_action( 'after_setup_theme', array( $this, 'colormag_post_meta_control_migrate' ) );
				//          add_action( 'after_setup_theme', array( $this, 'colormag_major_update_v1_customize_migrate' ) );
				add_action( 'after_setup_theme', array( $this, 'colormag_top_bar_options_migrate' ) );
				add_action( 'after_setup_theme', array( $this, 'colormag_breadcrumb_options_migrate' ) );
				add_action( 'after_setup_theme', array( $this, 'colormag_autoload_posts_control_migrate' ) );

				/**
				 * ColorMag revamp migrations.
				 */
				$this->old_theme_mods = get_theme_mods();
				add_action( 'after_setup_theme', array( $this, 'colormag_major_update_migration_v3' ) );
			}

			$enable_builder = get_theme_mod( 'colormag_enable_builder', '' );
			add_action( 'themegrill_ajax_demo_imported', [ $this, 'colormag_builder_migration' ], 25 );
			if ( $enable_builder ) {
				add_action( 'after_setup_theme', [ $this, 'colormag_builder_migration' ], 25 );
			}

			add_action( 'after_setup_theme', [ $this, 'colormag_outside_background_migration' ], 25 );
			add_action( 'themegrill_ajax_demo_imported', array( $this, 'colormag_color_preset' ), 25 );
			add_action( 'after_setup_theme', [ $this, 'colormag_logo_height_migration' ], 25 );

			$theme_installed_time = get_option( 'colormag_theme_installed_time' ); // timestamp
			$today                = strtotime( '2025-09-22' );

			if ( ! colormag_fresh_install() ) {
				add_action( 'after_setup_theme', [ $this, 'colormag_container_sidebar_migration' ], 25 );
				add_action( 'after_setup_theme', [ $this, 'colormag_typography_migration' ], 30 );
			}
			add_action( 'themegrill_ajax_demo_imported', [ $this, 'colormag_container_sidebar_migration' ], 25 );

			add_action( 'after_setup_theme', [ $this, 'colormag_headings_colors_to_typography' ], 26 );
		}

		public function colormag_headings_colors_to_typography() {

			if ( get_option( 'colormag_headings_colors_to_typography_migration' ) ) {
				return;
			}

			$headings_typography = get_theme_mod(
				'colormag_headings_typography',
				array(
					'font-family'    => 'inherit',
					'font-weight'    => 'regular',
					'subsets'        => array( 'latin' ),
					'font-size'      => array(
						'desktop' => array(
							'size' => '15',
							'unit' => 'px',
						),
						'tablet'  => array(
							'size' => '',
							'unit' => 'px',
						),
						'mobile'  => array(
							'size' => '',
							'unit' => 'px',
						),
					),
					'line-height'    => array(
						'desktop' => array(
							'size' => '1.6',
							'unit' => '-',
						),
						'tablet'  => array(
							'size' => '',
							'unit' => '-',
						),
						'mobile'  => array(
							'size' => '',
							'unit' => '-',
						),
					),
					'letter-spacing' => array(
						'desktop' => array(
							'size' => '',
							'unit' => 'px',
						),
						'tablet'  => array(
							'size' => '',
							'unit' => 'px',
						),
						'mobile'  => array(
							'size' => '',
							'unit' => 'px',
						),
					),
				)
			);
			$h1_typography       = get_theme_mod(
				'colormag_h1_typography',
				array(
					'font-family'    => 'inherit',
					'font-weight'    => 'Inherit',
					'subsets'        => array( 'latin' ),
					'font-size'      => array(
						'desktop' => array(
							'size' => '36',
							'unit' => 'px',
						),
						'tablet'  => array(
							'size' => '',
							'unit' => 'px',
						),
						'mobile'  => array(
							'size' => '',
							'unit' => 'px',
						),
					),
					'line-height'    => array(
						'desktop' => array(
							'size' => '1.2',
							'unit' => '-',
						),
						'tablet'  => array(
							'size' => '',
							'unit' => '',
						),
						'mobile'  => array(
							'size' => '',
							'unit' => '',
						),
					),
					'font-style'     => 'normal',
					'text-transform' => 'none',
					'color'          => 'var(--cm-color-6)',
				)
			);
			$h2_typography       = get_theme_mod(
				'colormag_h2_typography',
				array(
					'font-family'    => 'inherit',
					'font-weight'    => 'Inherit',
					'subsets'        => array( 'latin' ),
					'font-size'      => array(
						'desktop' => array(
							'size' => '32',
							'unit' => 'px',
						),
						'tablet'  => array(
							'size' => '',
							'unit' => '',
						),
						'mobile'  => array(
							'size' => '',
							'unit' => '',
						),
					),
					'line-height'    => array(
						'desktop' => array(
							'size' => '1.2',
							'unit' => '-',
						),
						'tablet'  => array(
							'size' => '',
							'unit' => '',
						),
						'mobile'  => array(
							'size' => '',
							'unit' => '',
						),
					),
					'font-style'     => 'normal',
					'text-transform' => 'none',
					'color'          => 'var(--cm-color-6)',
				)
			);
			$h3_typography       = get_theme_mod(
				'colormag_h3_typography',
				array(
					'font-family'    => 'inherit',
					'font-weight'    => 'Inherit',
					'subsets'        => array( 'latin' ),
					'font-size'      => array(
						'desktop' => array(
							'size' => '24',
							'unit' => 'px',
						),
						'tablet'  => array(
							'size' => '',
							'unit' => '',
						),
						'mobile'  => array(
							'size' => '',
							'unit' => '',
						),
					),
					'line-height'    => array(
						'desktop' => array(
							'size' => '1.2',
							'unit' => '-',
						),
						'tablet'  => array(
							'size' => '',
							'unit' => '',
						),
						'mobile'  => array(
							'size' => '',
							'unit' => '',
						),
					),
					'font-style'     => 'normal',
					'text-transform' => 'none',
					'color'          => 'var(--cm-color-6)',
				)
			);
			$headings_color      = get_theme_mod( 'colormag_headings_color', 'var(--cm-color-6)' );
			$h1_color            = get_theme_mod( 'colormag_h1_color', 'var(--cm-color-6)' );
			$h2_color            = get_theme_mod( 'colormag_h2_color', 'var(--cm-color-6)' );
			$h3_color            = get_theme_mod( 'colormag_h3_color', 'var(--cm-color-6)' );

			$headings_typography['color'] = $headings_color;
			set_theme_mod( 'colormag_headings_typography', $headings_typography );

			$h1_typography['color'] = $h1_color;
			set_theme_mod( 'colormag_h1_typography', $h1_typography );

			$h2_typography['color'] = $h2_color;
			set_theme_mod( 'colormag_h2_typography', $h2_typography );

			$h3_typography['color'] = $h3_color;
			set_theme_mod( 'colormag_h3_typography', $h3_typography );

			update_option( 'colormag_headings_colors_to_typography_migration', true );
		}

		public function colormag_typography_migration() {

			if ( get_option( 'colormag_typography_migration' ) ) {
				return;
			}

			// Default values for comparison
			$default_typography_presets   = '';
			$default_base_typography_body = array(
				'font-family'    => 'inherit',
				'font-weight'    => 'regular',
				'subsets'        => array( 'latin' ),
				'font-size'      => array(
					'desktop' => array(
						'size' => '15',
						'unit' => 'px',
					),
					'tablet'  => array(
						'size' => '',
						'unit' => 'px',
					),
					'mobile'  => array(
						'size' => '',
						'unit' => 'px',
					),
				),
				'line-height'    => array(
					'desktop' => array(
						'size' => '1.6',
						'unit' => '-',
					),
					'tablet'  => array(
						'size' => '',
						'unit' => '-',
					),
					'mobile'  => array(
						'size' => '',
						'unit' => '-',
					),
				),
				'letter-spacing' => array(
					'desktop' => array(
						'size' => '',
						'unit' => 'px',
					),
					'tablet'  => array(
						'size' => '',
						'unit' => 'px',
					),
					'mobile'  => array(
						'size' => '',
						'unit' => 'px',
					),
				),
			);

			$default_base_heading_typography = array(
				'font-family'    => 'inherit',
				'font-weight'    => 'regular',
				'subsets'        => array( 'latin' ),
				'line-height'    => array(
					'desktop' => array(
						'size' => '1.2',
						'unit' => '-',
					),
					'tablet'  => array(
						'size' => '',
						'unit' => '',
					),
					'mobile'  => array(
						'size' => '',
						'unit' => '',
					),
				),
				'letter-spacing' => array(
					'desktop' => array(
						'size' => '',
						'unit' => 'px',
					),
					'tablet'  => array(
						'size' => '',
						'unit' => 'px',
					),
					'mobile'  => array(
						'size' => '',
						'unit' => 'px',
					),
				),
				'font-style'     => 'normal',
				'text-transform' => 'none',
			);

			// Get current values
			$current_base_typography_body    = get_theme_mod( 'colormag_base_typography', $default_base_typography_body );
			$current_base_heading_typography = get_theme_mod( 'colormag_headings_typography', $default_base_heading_typography );

			// Check if current values are different from default values
			$should_migrate = false;

			// Check base typography body
			if ( $current_base_typography_body !== $default_base_typography_body ) {
				$should_migrate = true;
			}

			// Check base heading typography
			if ( $current_base_heading_typography !== $default_base_heading_typography ) {
				$should_migrate = true;
			}

			// Only run migration if current values are different from default values
			if ( ! $should_migrate ) {
				return;
			}

			remove_theme_mod( 'colormag_typography_presets' );

			$base_typography = get_theme_mod(
				'colormag_base_typography_body',
				$default_base_typography_body
			);

			set_theme_mod( 'colormag_base_typography_body', $base_typography );

			$base_heading_typography = get_theme_mod(
				'colormag_base_typography_heading',
				$default_base_heading_typography
			);

			set_theme_mod( 'colormag_base_typography_heading', $base_heading_typography );

			update_option( 'colormag_typography_migration', true );
		}

		public function colormag_container_sidebar_migration() {

			$should_run    = false;
			$color_palette = get_theme_mod( 'colormag_color_palette', '' );
			if ( ! empty( $color_palette ) && isset( $color_palette['updated_at'] ) ) {
				$should_run = false;
			} elseif ( doing_action( 'themegrill_ajax_demo_imported' ) ) {
				$should_run = true;
				set_theme_mod( 'colormag_enable_site_identity', false );
			} elseif ( ! get_option( 'colormag_container_sidebar_migration' ) ) {
				$should_run = true;
			}

			if ( ! $should_run ) {
				return;
			}

			$colormag_default_sidebar_layout    = get_theme_mod( 'colormag_default_sidebar_layout', 'right_sidebar' );
			$colormag_page_layout               = get_theme_mod( 'colormag_page_sidebar_layout', 'right_sidebar' );
			$colormag_post_layout               = get_theme_mod( 'colormag_post_sidebar_layout', 'right_sidebar' );
			$colormag_post_meta_date_style      = get_theme_mod( 'colormag_post_meta_date_style', 'style-1' );
			$colormag_post_meta_author_style    = get_theme_mod( 'colormag_post_meta_author_style', 'style-1' );
			$colormag_post_meta_color           = get_theme_mod( 'colormag_post_meta_color', '' );
			$colormag_blog_post_meta_typography = get_theme_mod( 'colormag_blog_post_meta_typography', '' );
			$post_meta_structure                = get_theme_mod(
				'colormag_post_meta_structure',
				array(
					'categories',
					'date',
					'author',
				)
			);
			if ( 'right_sidebar' === $colormag_default_sidebar_layout || 'left_sidebar' === $colormag_default_sidebar_layout || 'two_sidebars' === $colormag_default_sidebar_layout ) {
				set_theme_mod( 'colormag_blog_sidebar_layout', $colormag_default_sidebar_layout );
				set_theme_mod( 'colormag_blog_container_layout', 'no_sidebar_full_width' );
			} elseif ( 'no_sidebar_full_width' === $colormag_default_sidebar_layout || 'no_sidebar_content_centered' === $colormag_default_sidebar_layout || 'no_sidebar_content_stretched' === $colormag_default_sidebar_layout ) {
				set_theme_mod( 'colormag_blog_container_layout', $colormag_default_sidebar_layout );
				set_theme_mod( 'colormag_blog_sidebar_layout', 'no_sidebar' );
			}

			if ( 'right_sidebar' === $colormag_page_layout || 'left_sidebar' === $colormag_page_layout || 'two_sidebars' === $colormag_page_layout ) {
				set_theme_mod( 'colormag_single_page_sidebar_layout', $colormag_page_layout );
				set_theme_mod( 'colormag_single_page_container_layout', 'no_sidebar_full_width' );
			} elseif ( 'no_sidebar_full_width' === $colormag_page_layout || 'no_sidebar_content_centered' === $colormag_page_layout || 'no_sidebar_content_stretched' === $colormag_page_layout ) {
				set_theme_mod( 'colormag_single_page_container_layout', $colormag_page_layout );
				set_theme_mod( 'colormag_single_page_sidebar_layout', 'no_sidebar' );
			}

			if ( 'right_sidebar' === $colormag_post_layout || 'left_sidebar' === $colormag_post_layout || 'two_sidebars' === $colormag_post_layout ) {
				set_theme_mod( 'colormag_single_post_sidebar_layout', $colormag_post_layout );
				set_theme_mod( 'colormag_single_post_container_layout', 'no_sidebar_full_width' );
			} elseif ( 'no_sidebar_full_width' === $colormag_post_layout || 'no_sidebar_content_centered' === $colormag_post_layout || 'no_sidebar_content_stretched' === $colormag_post_layout ) {
				set_theme_mod( 'colormag_single_post_container_layout', $colormag_post_layout );
				set_theme_mod( 'colormag_single_post_sidebar_layout', 'no_sidebar' );
			}

			set_theme_mod( 'colormag_search_page_meta_structure', $post_meta_structure );
			set_theme_mod( 'colormag_single_post_meta_structure', $post_meta_structure );
			set_theme_mod( 'colormag_blog_post_meta_structure', $post_meta_structure );

			set_theme_mod( 'colormag_blog_post_meta_date_style', $colormag_post_meta_date_style );
			set_theme_mod( 'colormag_single_post_meta_date_style', $colormag_post_meta_date_style );
			set_theme_mod( 'colormag_search_page_meta_date_style', $colormag_post_meta_date_style );

			set_theme_mod( 'colormag_blog_post_meta_author_style', $colormag_post_meta_author_style );
			set_theme_mod( 'colormag_single_post_meta_author_style', $colormag_post_meta_author_style );
			set_theme_mod( 'colormag_search_page_meta_author_style', $colormag_post_meta_author_style );

			set_theme_mod( 'colormag_blog_post_meta_color', $colormag_post_meta_color );
			set_theme_mod( 'colormag_blog_post_meta_typography', $colormag_blog_post_meta_typography );
			set_theme_mod( 'colormag_single_post_meta_color', $colormag_post_meta_color );
			set_theme_mod( 'colormag_single_post_meta_typography', $colormag_blog_post_meta_typography );
			set_theme_mod( 'colormag_search_page_meta_color', $colormag_post_meta_color );
			set_theme_mod( 'colormag_search_page_meta_typography', $colormag_blog_post_meta_typography );

			// Post meta migration - Process in batches to avoid memory exhaustion.
			// Reduced batch size to prevent memory issues with meta operations
			$batch_size = 25;
			$offset     = 0;

			while ( true ) {
				// Check available memory before processing batch
				$memory_usage = memory_get_usage( true );
				$memory_limit = ini_get( 'memory_limit' );
				// Convert memory limit to bytes (fallback if wp_convert_hr_to_bytes doesn't exist)
				if ( function_exists( 'wp_convert_hr_to_bytes' ) ) {
					$memory_limit_bytes = wp_convert_hr_to_bytes( $memory_limit );
				} else {
					// Manual conversion
					$memory_limit_bytes = $memory_limit;
					if ( preg_match( '/^(\d+)(.)$/', $memory_limit, $matches ) ) {
						$number = $matches[1];
						$unit   = strtolower( $matches[2] );
						switch ( $unit ) {
							case 'g':
								$memory_limit_bytes = $number * 1024 * 1024 * 1024;
								break;
							case 'm':
								$memory_limit_bytes = $number * 1024 * 1024;
								break;
							case 'k':
								$memory_limit_bytes = $number * 1024;
								break;
						}
					}
				}
				// If memory usage is above 80%, skip this batch and try again later
				if ( $memory_limit_bytes > 0 && $memory_usage > ( $memory_limit_bytes * 0.8 ) ) {
					// Set a flag to retry later
					update_option( 'colormag_container_sidebar_migration_pending', true );
					break;
				}
				
				$args = [
					'post_type'              => 'any',
					'posts_per_page'         => $batch_size,
					'offset'                 => $offset,
					'fields'                 => 'ids', // Only get post IDs to reduce memory usage.
					'no_found_rows'          => true,
					'update_post_meta_cache' => false,
					'update_post_term_cache' => false,
					'cache_results'          => false, // Disable query caching
				];

				$the_query = new WP_Query( $args );

				// Break if no more posts.
				if ( ! $the_query->have_posts() ) {
					wp_reset_postdata();
					$the_query = null;
					break;
				}

				// Process each post ID.
				$post_ids = $the_query->posts;
				foreach ( $post_ids as $post_id ) {
					// Use direct database query to avoid meta cache
					global $wpdb;
					$colormag_post_layout = $wpdb->get_var( $wpdb->prepare(
						"SELECT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = 'colormag_page_layout' LIMIT 1",
						$post_id
					) );

					if ( 'right_sidebar' === $colormag_post_layout || 'left_sidebar' === $colormag_post_layout || 'two_sidebars' === $colormag_post_layout ) {
						update_post_meta( $post_id, 'colormag_page_sidebar_layout', $colormag_post_layout );
						update_post_meta( $post_id, 'colormag_page_container_layout', 'no_sidebar_full_width' );
					}
					if ( 'no_sidebar_full_width' === $colormag_post_layout || 'no_sidebar_content_centered' === $colormag_post_layout || 'no_sidebar_content_stretched' === $colormag_post_layout ) {
						update_post_meta( $post_id, 'colormag_page_container_layout', $colormag_post_layout );
						update_post_meta( $post_id, 'colormag_page_sidebar_layout', 'no_sidebar' );
					}

					if ( 'default_layout' === $colormag_post_layout ) {
						update_post_meta( $post_id, 'colormag_page_container_layout', $colormag_post_layout );
						update_post_meta( $post_id, 'colormag_page_sidebar_layout', $colormag_post_layout );
					}
					
					// Clear post meta cache for this post
					wp_cache_delete( $post_id, 'post_meta' );
				}
				
				// Clear variables
				$post_ids = null;
				unset( $post_ids );

				// Safety check: if we got fewer posts than batch size, we're done.
				// Store count before resetting query
				$post_count = $the_query->post_count;

				wp_reset_postdata();
				
				// Clear query cache to free memory
				$the_query = null;
				unset( $the_query );

				// Move to next batch.
				$offset += $batch_size;
				
				// Clear WordPress object cache
				wp_cache_flush();
				
				// Force garbage collection to free memory
				if ( function_exists( 'gc_collect_cycles' ) ) {
					gc_collect_cycles();
				}
				
				if ( $post_count < $batch_size ) {
					break;
				}
			}

			$default_palette = array(
				'id'     => 'preset-5',
				'name'   => 'Preset 5',
				'colors' => array(
					'colormag-color-1' => '#FFFFFF',
					'colormag-color-2' => '#FAFAFA',
					'colormag-color-4' => '#d4d4d8',
					'colormag-color-5' => '#E4E4E7',
					'colormag-color-3' => '#3F3F46',
					'colormag-color-6' => '#27272a',
					'colormag-color-7' => '#333333',
					'colormag-color-8' => '#444444',
					'colormag-color-9' => '#207daf',
				),
			);

			$color_palette = get_theme_mod(
				'colormag_color_palette',
				$default_palette
			);

			// Check if color_palette has colors and is properly structured
			if ( ! empty( $color_palette ) && is_array( $color_palette ) && isset( $color_palette['colors'] ) && is_array( $color_palette['colors'] ) && ! empty( $color_palette['colors'] ) ) {
				$colors = $color_palette['colors'];
			} else {
				$colors = $default_palette['colors'];
			}

			$color_id = array(
				array( 'colormag_primary_color', 'var(--colormag-color-9)', '#207daf' ),
				array( 'colormag_base_color', 'var(--colormag-color-8)', '#444444' ),
				array( 'colormag_headings_color', 'var(--colormag-color-7)', '#333333' ),
				array( 'colormag_h1_color', '', '' ),
				array( 'colormag_h2_color', 'var(--colormag-color-7)', '#333333' ),
				array( 'colormag_h3_color', 'var(--colormag-color-7)', '#333333' ),
				array( 'colormag_link_color', 'var(--colormag-color-9)', '#207daf' ),
				array( 'colormag_link_hover_color', 'var(--colormag-color-9)', '#207daf' ),
				array( 'colormag_sidebar_widget_title_color', '', '#ffffff' ),
				array( 'colormag_button_color', '', '#ffffff' ),
				array( 'colormag_button_hover_color', '', '' ),
				array( 'colormag_button_background_color', 'var(--colormag-color-9)', '#207daf' ),
				array( 'colormag_button_background_hover_color', '', '' ),
				array( 'colormag_button_border_color', '', '' ),
				array( 'colormag_button_border_hover_color', '', '' ),
				array( 'colormag_widget_view_all_button_color', '', '' ),
				array( 'colormag_widget_view_all_button_background', '', '' ),
				array( 'colormag_form_element_text_color', '' ),
				array( 'colormag_form_element_text_focus_color', '', '' ),
				array( 'colormag_form_element_background_color', '', '' ),
				array( 'colormag_form_element_background_focus_color', '', '' ),
				array( 'colormag_form_element_border_color', '', '' ),
				array( 'colormag_global_accessibility_color', '', '' ),
				array( 'colormag_input_accessibility_color', '', '' ),
				array( 'colormag_header_site_identity_color', 'var(--colormag-color-9)', '#207daf' ),
				array( 'colormag_header_site_identity_hover_color', '', '' ),
				array( 'colormag_header_site_tagline_color', '', '' ),
				array( 'colormag_header_primary_menu_text_color', '', '' ),
				array( 'colormag_header_primary_menu_selected_hovered_text_color', '', '' ),
				array( 'colormag_header_primary_menu_active_text_color', '', '' ),
				array( 'colormag_header_primary_menu_hover_background', '', '' ),
				array( 'colormag_header_primary_menu_active_background', '', '' ),
				array( 'colormag_header_secondary_menu_text_color', '', '' ),
				array( 'colormag_header_secondary_menu_selected_hovered_text_color', '', '' ),
				array( 'colormag_header_secondary_menu_hover_background', '', '' ),
				array( 'colormag_header_button_color', '', '#ffffff' ),
				array( 'colormag_header_button_hover_color', '', '#ffffff' ),
				array( 'colormag_header_button_background_color', 'var(--colormag-color-9)', '#207daf' ),
				array( 'colormag_header_button_background_hover_color', '', '#ffffff' ),
				array( 'colormag_header_search_text_color', '', '' ),
				array( 'colormag_header_search_placeholder_color', '', '' ),
				array( 'colormag_header_search_background', '', '' ),
				array( 'colormag_header_search_border_color', '', '' ),
				array( 'colormag_header_search_icon_color', '', '' ),
				array( 'colormag_header_search_button_background', '', '' ),
				array( 'colormag_header_search_button_hover_background', '', '' ),
				array( 'colormag_header_html_1_text_color', '', '' ),
				array( 'colormag_header_html_1_link_color', '', '' ),
				array( 'colormag_header_html_1_link_hover_color', '', '' ),
				array( 'colormag_widget_1_title_color', '', '' ),
				array( 'colormag_widget_1_link_color', '', '' ),
				array( 'colormag_widget_1_content_color', '', '' ),
				array( 'colormag_widget_2_title_color', '', '' ),
				array( 'colormag_widget_2_link_color', '', '' ),
				array( 'colormag_widget_2_content_color', '', '' ),
				array( 'colormag_header_socials_color', '', '' ),
				array( 'colormag_date_color', '', '' ),
				array( 'colormag_news_ticker_color', '', '' ),
				array( 'colormag_news_ticker_link_color', '', '' ),
				array( 'colormag_header_random_icon_color', '', '' ),
				array( 'colormag_header_random_icon_hover_color', '', '' ),
				array( 'colormag_header_home_icon_color', '', '' ),
				array( 'colormag_header_home_icon_background_color', '', '' ),
				array( 'colormag_header_home_icon_hover_background_color', '', '' ),
				array( 'colormag_off_canvas_text_color', '', '' ),
				array( 'colormag_off_canvas_border_color', '', '' ),
				array( 'colormag_post_title_color', '', '#333333' ),
				array( 'colormag_page_title_color', '', '#333333' ),
				array( 'colormag_footer_html_1_text_color', '', '' ),
				array( 'colormag_footer_html_1_link_color', '', '' ),
				array( 'colormag_footer_html_1_link_hover_color', '', '' ),
				array( 'colormag_footer_widget_1_title_color', '', '' ),
				array( 'colormag_footer_widget_1_link_color', '', '' ),
				array( 'colormag_footer_widget_1_link_hover_color', '', '' ),
				array( 'colormag_footer_widget_1_content_color', '', '' ),
				array( 'colormag_footer_widget_2_title_color', '', '' ),
				array( 'colormag_footer_widget_2_link_color', '', '' ),
				array( 'colormag_footer_widget_2_link_hover_color', '', '' ),
				array( 'colormag_footer_widget_2_content_color', '', '' ),
				array( 'colormag_footer_widget_3_title_color', '', '' ),
				array( 'colormag_footer_widget_3_link_color', '', '' ),
				array( 'colormag_footer_widget_3_link_hover_color', '', '' ),
				array( 'colormag_footer_widget_3_content_color', '', '' ),
				array( 'colormag_footer_widget_4_title_color', '', '' ),
				array( 'colormag_footer_widget_4_link_color', '', '' ),
				array( 'colormag_footer_widget_4_link_hover_color', '', '' ),
				array( 'colormag_footer_widget_4_content_color', '', '' ),
				array( 'colormag_footer_widget_5_title_color', '', '' ),
				array( 'colormag_footer_widget_5_link_color', '', '' ),
				array( 'colormag_footer_widget_5_link_hover_color', '', '' ),
				array( 'colormag_footer_widget_5_content_color', '', '' ),
				array( 'colormag_footer_widget_6_title_color', '', '' ),
				array( 'colormag_footer_widget_6_link_color', '', '' ),
				array( 'colormag_footer_widget_6_link_hover_color', '', '' ),
				array( 'colormag_footer_widget_6_content_color', '', '' ),
				array( 'colormag_footer_widget_7_title_color', '', '' ),
				array( 'colormag_footer_widget_7_link_color', '', '' ),
				array( 'colormag_footer_widget_7_link_hover_color', '', '' ),
				array( 'colormag_footer_widget_7_content_color', '', '' ),
				array( 'colormag_footer_menu_color', '', '' ),
				array( 'colormag_footer_menu_hover_color', '', '' ),
				array( 'colormag_footer_button_color', '', '#ffffff' ),
				array( 'colormag_footer_button_hover_color', '', '#ffffff' ),
				array( 'colormag_footer_button_background_color', '', '#027abb' ),
				array( 'colormag_footer_button_background_hover_color', '', '#ffffff' ),
				array( 'colormag_footer_button_border_color', '', '#ffffff' ),
				array( 'colormag_footer_copyright_text_color', '', '' ),
				array( 'colormag_footer_copyright_link_color', '', '' ),
				array( 'colormag_footer_copyright_link_hover_color', '', '' ),
				array( 'colormag_footer_bottom_area_color', '', '#fafafa' ),
				array( 'colormag_footer_bottom_area_border_color', 'var(--colormag-color-3)', '#3F3F46' ),
				array( 'colormag_footer_main_area_color', '', '' ),
				array( 'colormag_footer_main_area_link_color', '', '' ),
				array( 'colormag_footer_main_area_link_hover_color', '', '' ),
				array( 'colormag_footer_main_area_border_color', '', '' ),
				array( 'colormag_footer_widgets_title_color', '', '' ),
				array( 'colormag_footer_widgets_item_border_bottom_color', '', '' ),
				array( 'colormag_footer_top_area_color', '', '' ),
				array( 'colormag_footer_top_area_border_color', '', '' ),
				array( 'colormag_header_bottom_area_color', '', '' ),
				array( 'colormag_header_bottom_area_border_color', 'var(--colormag-color-9)', '#207daf' ),
				array( 'colormag_header_main_area_color', '', '' ),
				array( 'colormag_header_main_area_border_color', '', '' ),
				array( 'colormag_header_top_area_color', '', '' ),
				array( 'colormag_header_top_area_border_color', '', '' ),
			);

			// Set colors from the palette.
			if ( ! empty( $colors ) ) {
				foreach ( $color_id as $color_setting ) {
					$color_value = get_theme_mod( $color_setting[0], $color_setting[1] );

					if ( ! empty( $color_value ) ) {

						if ( strpos( $color_value, 'var(--colormag-color' ) === 0 ) {
							$key = str_replace( 'var(--', '', $color_value );
							$key = str_replace( ')', '', $key );
							if ( isset( $colors[ $key ] ) ) {
								set_theme_mod( $color_setting[0], $colors[ $key ] );
							}
						} else {
							// If the value is not a CSS variable, ensure it's saved as is.
							if ( ! empty( $color_value ) ) {
								set_theme_mod( $color_setting[0], $color_value );
							} elseif ( ! empty( $color_setting[2] ) ) {
								set_theme_mod( $color_setting[0], $color_setting[2] );
							}
						}
					}
				}
			}

			// Set background colors from the palette.
			if ( ! empty( $colors ) ) {
				$bg_id = array(
					array( 'colormag_header_primary_sub_menu_background', '#232323' ),
					array( 'colormag_header_top_area_background', '#f4f4f5' ),
					array( 'colormag_header_main_area_background', '#FAFAFA' ),
					array( 'colormag_header_bottom_area_background', 'var(--colormag-color-6)' ),
					array( 'colormag_footer_top_area_background', '' ),
					array( 'colormag_footer_main_area_background', '' ),
					array( 'colormag_footer_bottom_area_background', 'var(--colormag-color-6)' ),
					array( 'colormag_off_canvas_button_background', '' ),
					array( 'colormag_off_canvas_background', '' ),
					array( 'colormag_header_secondary_sub_menu_background', '#232323' ),
					array( 'colormag_inside_container_background', '#ffffff' ),
					array( 'colormag_outside_container_background', '' ),
				);
				foreach ( $bg_id as $color_setting ) {
					$value = get_theme_mod(
						$color_setting[0],
						array(
							'background-color'      => $color_setting[1],
							'background-image'      => '',
							'background-repeat'     => 'repeat',
							'background-position'   => 'center center',
							'background-size'       => 'contain',
							'background-attachment' => 'scroll',
						)
					);
					if ( is_array( $value ) && isset( $value['background-color'] ) ) {
						$color_value = $value['background-color'];
						if ( strpos( $color_value, 'var(--colormag-color' ) === 0 ) {
							$key = str_replace( 'var(--', '', $color_value );
							$key = str_replace( ')', '', $key );
							if ( isset( $colors[ $key ] ) ) {
								$value['background-color'] = $colors[ $key ];
								set_theme_mod( $color_setting[0], $value );
							}
						} elseif ( ! empty( $color_value ) ) {
							// If the value is not a CSS variable, ensure it's saved as is.
							$value['background-color'] = $color_value;
							set_theme_mod( $color_setting[0], $value );
						} elseif ( ! empty( $color_setting[1] ) ) {
							$value['background-color'] = $color_setting[1];
							set_theme_mod( $color_setting[0], $value );
						}
					}
				}
			}

			// Reset color palette if it contains old colors.
			$_colors_palette = get_theme_mod(
				'colormag_color_palette'
			);

			if ( isset( $_colors_palette['colors'] ) && array_key_exists( 'colormag-color-3', $_colors_palette['colors'] ) ) {
				set_theme_mod( 'colormag_color_palette', array() );
			}

			// Set dark skin to preset-6 if skin is dark.
			$skin_setting = get_theme_mod( 'colormag_color_skin_setting', 'white' );
			if ( 'dark' === $skin_setting ) {
				set_theme_mod( 'colormag_dark_skin', 'preset-6' );
			}

			update_option( 'colormag_container_sidebar_migration', true );
		}

		public function colormag_logo_height_migration() {
			$logo_height = get_theme_mod( 'colormag_header_site_logo_height' );

			if ( get_option( 'colormag_logo_height_migration' ) || isset( $logo_height['desktop'] ) || isset( $logo_height['tablet'] ) || isset( $logo_height['mobile'] ) ) {
				return;
			}

			if ( ! empty( $logo_height ) && is_array( $logo_height ) && isset( $logo_height['size'] ) && ! empty( $logo_height['size'] ) ) {
				$logo_height = $logo_height['size'];
				$value       = array(
					'desktop' => array(
						'size' => $logo_height,
						'unit' => 'px',
					),
					'tablet'  => array(
						'size' => '',
						'unit' => '',
					),
					'mobile'  => array(
						'size' => '',
						'unit' => '',
					),
				);
				set_theme_mod( 'colormag_header_site_logo_height', $value );
				update_option( 'colormag_logo_height_migration', true );
			}
		}

		public function colormag_color_preset() {

			$color_preset = get_theme_mod(
				'colormag_color_palette',
				array(
					'id'     => 'preset-5',
					'name'   => 'Preset 5',
					'colors' => array(
						'cm-color-3' => '#FFFFFF',
						'cm-color-4' => '#FAFAFA',
						'cm-color-9' => '#d4d4d8',
						'cm-color-8' => '#E4E4E7',
						'cm-color-2' => '#3F3F46',
						'cm-color-5' => '#27272a',
						'cm-color-7' => '#333333',
						'cm-color-6' => '#444444',
						'cm-color-1' => '#207daf',
					),
				)
			);
			if ( ! empty( $color_preset ) ) {
				$color_preset = json_decode( wp_json_encode( $color_preset ), true );
				if ( isset( $color_preset['id'] ) && 'preset-5' === $color_preset['id'] ) {
					set_theme_mod( 'colormag_color_palette', array() );
				}
			}
		}

		/**
		 * Migrate all the customize options for 3.0.0 theme update.
		 *
		 * @since ColorMag 3.0.0
		 */
		public function colormag_major_update_v1_customize_migrate() {

			$demo_import_migration = self::colormag_demo_import_migration();

			// Migrate the customize option if migration is done manually.
			if ( ! $demo_import_migration ) {

				// Bail out if the migration is already done.
				if ( get_option( 'colormag_major_update_v1_customize_migrate' ) ) {
					return;
				}
			}

			/**
			 * Assign the required variables.
			 */
			// Header.
			$header_background_color           = get_theme_mod( 'colormag_header_background_color', '#ffffff' );
			$site_title_font_family            = get_theme_mod( 'colormag_site_title_font', 'Open Sans' );
			$site_title_font_size              = get_theme_mod( 'colormag_title_font_size', '46' );
			$site_tagline_font_family          = get_theme_mod( 'colormag_site_tagline_font', 'Open Sans' );
			$site_tagline_font_size            = get_theme_mod( 'colormag_tagline_font_size', '16' );
			$primary_menu_background_color     = get_theme_mod( 'colormag_primary_menu_background_color', '#232323' );
			$primary_sub_menu_background_color = get_theme_mod( 'colormag_primary_sub_menu_background_color', '#232323' );
			$primary_menu_font_family          = get_theme_mod( 'colormag_primary_menu_font', 'Open Sans' );
			$primary_menu_font_size            = get_theme_mod( 'colormag_primary_menu_font_size', '14' );
			$primary_sub_menu_font_size        = get_theme_mod( 'colormag_primary_sub_menu_font_size', '14' );

			// Content.
			$post_title_font_size = get_theme_mod( 'colormag_post_title_font_size', '32' );
			$page_title_font_size = get_theme_mod( 'colormag_page_title_font_size', '34' );
			$post_meta_font_size  = get_theme_mod( 'colormag_post_meta_font_size', '12' );
			$button_font_size     = get_theme_mod( 'colormag_button_text_font_size', '12' );
			$post_content_color   = get_theme_mod( 'colormag_content_section_background_color', '#ffffff' );

			// Footer.
			$footer_background_image              = get_theme_mod( 'colormag_footer_background_image' );
			$footer_background_image_position     = get_theme_mod( 'colormag_footer_background_image_position', 'center-center' );
			$footer_background_size               = get_theme_mod( 'colormag_footer_background_image_size', 'auto' );
			$footer_background_attachment         = get_theme_mod( 'colormag_footer_background_image_attachment', 'scroll' );
			$footer_background_repeat             = get_theme_mod( 'colormag_footer_background_image_repeat', 'repeat' );
			$footer_copyright_background_color    = get_theme_mod( 'colormag_footer_copyright_part_background_color', '' );
			$footer_copyright_text_font_size      = get_theme_mod( 'colormag_footer_copyright_text_font_size', '14' );
			$footer_small_menu_font_size          = get_theme_mod( 'colormag_footer_small_menu_font_size', '14' );
			$footer_widget_background_color       = get_theme_mod( 'colormag_footer_widget_background_color', '' );
			$upper_footer_widget_background_color = get_theme_mod( 'colormag_upper_footer_widget_background_color', '#2c2e34' );

			// Typography.
			$content_font                    = get_theme_mod( 'colormag_content_font', 'Open Sans' );
			$content_font_size               = get_theme_mod( 'colormag_content_font_size', '15' );
			$all_titles_font                 = get_theme_mod( 'colormag_all_titles_font', 'Open Sans' );
			$heading_h1_font_size            = get_theme_mod( 'colormag_heading_h1_font_size', '36' );
			$heading_h2_font_size            = get_theme_mod( 'colormag_heading_h2_font_size', '32' );
			$heading_h3_font_size            = get_theme_mod( 'colormag_heading_h3_font_size', '28' );
			$heading_h4_font_size            = get_theme_mod( 'colormag_heading_h4_font_size', '24' );
			$heading_h5_font_size            = get_theme_mod( 'colormag_heading_h5_font_size', '22' );
			$heading_h6_font_size            = get_theme_mod( 'colormag_heading_h6_font_size', '18' );
			$widget_title_font_size          = get_theme_mod( 'colormag_widget_title_font_size', '18' );
			$comment_title_font_size         = get_theme_mod( 'colormag_comment_title_font_size', '24' );
			$footer_widget_title_font_size   = get_theme_mod( 'colormag_footer_widget_title_font_size', '18' );
			$footer_widget_content_font_size = get_theme_mod( 'colormag_footer_widget_content_font_size', '14' );

			/**
			 * Update the theme mods data.
			 */
			/**
			 * Header options.
			 */
			// Header background.
			if ( '#ffffff' !== $header_background_color ) {
				set_theme_mod(
					'colormag_header_background_setting',
					array(
						'background-color'      => $header_background_color,
						'background-image'      => '',
						'background-position'   => 'center center',
						'background-size'       => 'auto',
						'background-attachment' => 'scroll',
						'background-repeat'     => 'repeat',
					)
				);
			}

			// Site title.
			if ( 'Open Sans' !== $site_title_font_family || '46' !== $site_title_font_size ) {
				set_theme_mod(
					'colormag_site_title_typography_setting',
					array(
						'font-family' => $site_title_font_family,
						'font-size'   => array(
							'desktop' => $site_title_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Site tagline.
			if ( 'Open Sans' !== $site_tagline_font_family || '16' !== $site_tagline_font_size ) {
				set_theme_mod(
					'colormag_site_tagline_typography_setting',
					array(
						'font-family' => $site_tagline_font_family,
						'font-size'   => array(
							'desktop' => $site_tagline_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Primary menu background.
			if ( '#27272A' !== $primary_menu_background_color ) {
				set_theme_mod(
					'colormag_primary_menu_background_setting',
					array(
						'background-color'      => $primary_menu_background_color,
						'background-image'      => '',
						'background-position'   => 'center center',
						'background-size'       => 'auto',
						'background-attachment' => 'scroll',
						'background-repeat'     => 'repeat',
					)
				);
			}

			// Primary sub menu background.
			if ( '#232323' !== $primary_sub_menu_background_color ) {
				set_theme_mod(
					'colormag_primary_sub_menu_background_setting',
					array(
						'background-color'      => $primary_sub_menu_background_color,
						'background-image'      => '',
						'background-position'   => 'center center',
						'background-size'       => 'auto',
						'background-attachment' => 'scroll',
						'background-repeat'     => 'repeat',
					)
				);
			}

			// Primary menu fonts.
			if ( 'Open Sans' !== $primary_menu_font_family || '14' !== $primary_menu_font_size ) {
				set_theme_mod(
					'colormag_primary_menu_typography_setting',
					array(
						'font-family' => $primary_menu_font_family,
						'font-weight' => '400',
						'font-size'   => array(
							'desktop' => $primary_menu_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Primary sub menu fonts.
			if ( '14' !== $primary_sub_menu_font_size ) {
				set_theme_mod(
					'colormag_primary_sub_menu_typography_setting',
					array(
						'font-size' => array(
							'desktop' => $primary_sub_menu_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			/**
			 * Post/Page/Blog options.
			 */
			// Post title fonts.
			if ( '32' !== $post_title_font_size ) {
				set_theme_mod(
					'colormag_post_title_typography_setting',
					array(
						'font-size' => array(
							'desktop' => $post_title_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Page title fonts.
			if ( '34' !== $page_title_font_size ) {
				set_theme_mod(
					'colormag_page_title_typography_setting',
					array(
						'font-size' => array(
							'desktop' => $page_title_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Post meta fonts.
			if ( '12' !== $post_meta_font_size ) {
				set_theme_mod(
					'colormag_post_meta_typography_setting',
					array(
						'font-size' => array(
							'desktop' => $post_meta_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Button fonts.
			if ( '12' !== $button_font_size ) {
				set_theme_mod(
					'colormag_button_typography_setting',
					array(
						'font-family'    => 'Open Sans',
						'font-weight'    => 'regular',
						'subsets'        => array( 'latin' ),
						'font-size'      => array(
							'desktop' => $button_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
						'line-height'    => array(
							'desktop' => '',
							'tablet'  => '',
							'mobile'  => '',
						),
						'letter-spacing' => array(
							'desktop' => '',
							'tablet'  => '',
							'mobile'  => '',
						),
						'font-style'     => 'normal',
						'text-transform' => 'none',
					)
				);
			}

			// Post content background.
			if ( '#ffffff' !== $post_content_color ) {
				set_theme_mod(
					'colormag_inside_container_background',
					array(
						'background-color'      => $post_content_color,
						'background-image'      => '',
						'background-position'   => 'center center',
						'background-size'       => 'auto',
						'background-attachment' => 'scroll',
						'background-repeat'     => 'repeat',
					)
				);
			}

			/**
			 * Footer options.
			 */
			// Footer background.
			if ( $footer_background_image || 'center-center' !== $footer_background_image_position || 'auto' !== $footer_background_size || 'scroll' !== $footer_background_attachment || 'repeat' !== $footer_background_repeat ) {
				set_theme_mod(
					'colormag_footer_background_setting',
					array(
						'background-color'      => '',
						'background-image'      => $footer_background_image,
						'background-position'   => str_replace( '-', ' ', $footer_background_image_position ),
						'background-size'       => $footer_background_size,
						'background-attachment' => $footer_background_attachment,
						'background-repeat'     => $footer_background_repeat,
					)
				);
			}

			// Footer copyright background.
			if ( '' !== $footer_copyright_background_color ) {
				set_theme_mod(
					'colormag_footer_copyright_background_setting',
					array(
						'background-color'      => $footer_copyright_background_color,
						'background-image'      => '',
						'background-position'   => 'center center',
						'background-size'       => 'auto',
						'background-attachment' => 'scroll',
						'background-repeat'     => 'repeat',
					)
				);
			}

			// Footer copyright fonts.
			if ( '14' !== $footer_copyright_text_font_size ) {
				set_theme_mod(
					'colormag_footer_copyright_typography_setting',
					array(
						'font-size' => array(
							'desktop' => $footer_copyright_text_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Footer menu fonts.
			if ( '14' !== $footer_small_menu_font_size ) {
				set_theme_mod(
					'colormag_footer_menu_typography_setting',
					array(
						'font-size' => array(
							'desktop' => $footer_small_menu_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Footer sidebar area background.
			if ( '' !== $footer_widget_background_color ) {
				set_theme_mod(
					'colormag_footer_sidebar_area_background_setting',
					array(
						'background-color'      => $footer_widget_background_color,
						'background-image'      => '',
						'background-position'   => 'center center',
						'background-size'       => 'auto',
						'background-attachment' => 'scroll',
						'background-repeat'     => 'repeat',
					)
				);
			}

			// Upper footer sidebar area background.
			if ( '#2c2e34' !== $upper_footer_widget_background_color ) {
				set_theme_mod(
					'colormag_footer_upper_sidebar_area_background_setting',
					array(
						'background-color'      => $upper_footer_widget_background_color,
						'background-image'      => '',
						'background-position'   => 'center center',
						'background-size'       => 'auto',
						'background-attachment' => 'scroll',
						'background-repeat'     => 'repeat',
					)
				);
			}

			/**
			 * Typography options.
			 */
			// Base fonts.
			if ( 'Open Sans' !== $content_font || '15' !== $content_font_size ) {
				set_theme_mod(
					'colormag_base_typography_setting',
					array(
						'font-family'    => $content_font,
						'font-weight'    => 'regular',
						'subsets'        => array( 'latin' ),
						'font-size'      => array(
							'desktop' => $content_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
						'line-height'    => array(
							'desktop' => '1.6',
							'tablet'  => '',
							'mobile'  => '',
						),
						'letter-spacing' => array(
							'desktop' => '',
							'tablet'  => '',
							'mobile'  => '',
						),
						'font-style'     => 'normal',
						'text-transform' => 'none',
					)
				);
			}

			// All title fonts.
			if ( 'Open Sans' !== $all_titles_font ) {
				set_theme_mod(
					'colormag_headings_typography_setting',
					array(
						'font-family'    => $all_titles_font,
						'font-weight'    => 'regular',
						'subsets'        => array( 'latin' ),
						'line-height'    => array(
							'desktop' => '1.2',
							'tablet'  => '',
							'mobile'  => '',
						),
						'letter-spacing' => array(
							'desktop' => '',
							'tablet'  => '',
							'mobile'  => '',
						),
						'font-style'     => 'normal',
						'text-transform' => 'none',
					)
				);
			}

			// Heading H1 fonts.
			if ( '36' !== $heading_h1_font_size ) {
				set_theme_mod(
					'colormag_h1_typography_setting',
					array(
						'font-family'    => 'Open Sans',
						'font-weight'    => 'regular',
						'subsets'        => array( 'latin' ),
						'font-size'      => array(
							'desktop' => $heading_h1_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
						'line-height'    => array(
							'desktop' => '1.2',
							'tablet'  => '',
							'mobile'  => '',
						),
						'letter-spacing' => array(
							'desktop' => '',
							'tablet'  => '',
							'mobile'  => '',
						),
						'font-style'     => 'normal',
						'text-transform' => 'none',
					)
				);
			}

			// Heading H2 fonts.
			if ( '32' !== $heading_h2_font_size ) {
				set_theme_mod(
					'colormag_h2_typography_setting',
					array(
						'font-family'    => 'Open Sans',
						'font-weight'    => 'regular',
						'subsets'        => array( 'latin' ),
						'font-size'      => array(
							'desktop' => $heading_h2_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
						'line-height'    => array(
							'desktop' => '1.2',
							'tablet'  => '',
							'mobile'  => '',
						),
						'letter-spacing' => array(
							'desktop' => '',
							'tablet'  => '',
							'mobile'  => '',
						),
						'font-style'     => 'normal',
						'text-transform' => 'none',
					)
				);
			}

			// Heading H3 fonts.
			if ( '28' !== $heading_h3_font_size ) {
				set_theme_mod(
					'colormag_h3_typography_setting',
					array(
						'font-family'    => 'Open Sans',
						'font-weight'    => 'regular',
						'subsets'        => array( 'latin' ),
						'font-size'      => array(
							'desktop' => $heading_h3_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
						'line-height'    => array(
							'desktop' => '1.2',
							'tablet'  => '',
							'mobile'  => '',
						),
						'letter-spacing' => array(
							'desktop' => '',
							'tablet'  => '',
							'mobile'  => '',
						),
						'font-style'     => 'normal',
						'text-transform' => 'none',
					)
				);
			}

			// Heading H4 fonts.
			if ( '24' !== $heading_h4_font_size ) {
				set_theme_mod(
					'colormag_h4_typography_setting',
					array(
						'font-size'   => array(
							'desktop' => $heading_h4_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
						'line-height' => array(
							'desktop' => '1.2',
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Heading H5 fonts.
			if ( '22' !== $heading_h5_font_size ) {
				set_theme_mod(
					'colormag_h5_typography',
					array(
						'font-size'   => array(
							'desktop' => $heading_h5_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
						'line-height' => array(
							'desktop' => '1.2',
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Heading H6 fonts.
			if ( '18' !== $heading_h6_font_size ) {
				set_theme_mod(
					'colormag_h6_typography_setting',
					array(
						'font-size'   => array(
							'desktop' => $heading_h6_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
						'line-height' => array(
							'desktop' => '1.2',
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Widget title fonts.
			if ( '18' !== $widget_title_font_size ) {
				set_theme_mod(
					'colormag_widget_title_typography_setting',
					array(
						'font-size' => array(
							'desktop' => $widget_title_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Comment title fonts.
			if ( '24' !== $comment_title_font_size ) {
				set_theme_mod(
					'colormag_comment_title_typography_setting',
					array(
						'font-size' => array(
							'desktop' => $comment_title_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Footer widget title fonts.
			if ( '18' !== $footer_widget_title_font_size ) {
				set_theme_mod(
					'colormag_footer_widget_title_typography_setting',
					array(
						'font-size' => array(
							'desktop' => $footer_widget_title_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			// Footer widget content fonts.
			if ( '14' !== $footer_widget_content_font_size ) {
				set_theme_mod(
					'colormag_footer_widget_content_typography_setting',
					array(
						'font-size' => array(
							'desktop' => $footer_widget_content_font_size,
							'tablet'  => '',
							'mobile'  => '',
						),
					)
				);
			}

			/**
			 * Remove unnecessary theme mods data.
			 */
			$remove_theme_mod_settings = array(

				// Header options.
				'colormag_site_title_font',
				'colormag_title_font_size',
				'colormag_site_tagline_font',
				'colormag_tagline_font_size',
				'colormag_header_background_color',
				'colormag_primary_menu_background_color',
				'colormag_primary_sub_menu_background_color',
				'colormag_primary_menu_font',
				'colormag_primary_menu_font_size',
				'colormag_primary_sub_menu_font_size',

				// Post/Page/Blog options.
				'colormag_post_title_font_size',
				'colormag_page_title_font_size',
				'colormag_post_meta_font_size',
				'colormag_button_text_font_size',
				'colormag_content_section_background_color',

				// Footer options.
				'colormag_footer_background_image',
				'colormag_footer_background_image_position',
				'colormag_footer_background_image_size',
				'colormag_footer_background_image_attachment',
				'colormag_footer_background_image_repeat',
				'colormag_footer_copyright_part_background_color',
				'colormag_footer_copyright_text_font_size',
				'colormag_footer_small_menu_font_size',
				'colormag_footer_widget_background_color',
				'colormag_upper_footer_widget_background_color',

				// Typography options.
				'colormag_content_font',
				'colormag_content_font_size',
				'colormag_all_titles_font',
				'colormag_heading_h1_font_size',
				'colormag_heading_h2_font_size',
				'colormag_heading_h3_font_size',
				'colormag_heading_h4_font_size',
				'colormag_heading_h5_font_size',
				'colormag_heading_h6_font_size',
				'colormag_widget_title_font_size',
				'colormag_comment_title_font_size',
				'colormag_footer_widget_title_font_size',
				'colormag_footer_widget_content_font_size',

			);

			// Loop through the theme mods to remove them.
			foreach ( $remove_theme_mod_settings as $remove_theme_mod_setting ) {
				remove_theme_mod( $remove_theme_mod_setting );
			}

			// Set flag to not repeat the migration process, ie, run it only once.
			update_option( 'colormag_major_update_v1_customize_migrate', true );

			// Set flag for demo import migration to not repeat the migration process, ie, run it only once.
			if ( $demo_import_migration ) {
				update_option( 'colormag_demo_import_migration_notice_dismiss', true );
			}
		}

		public function colormag_social_icons_control_migrate() {

			$social_icon            = get_theme_mod( 'colormag_social_link_activate', 0 );
			$social_icon_visibility = get_theme_mod( 'colormag_social_link_location_option', 'both' );

			// Disable social icon on header if enabled on footer only.
			if ( 0 !== $social_icon ) {
				set_theme_mod( 'colormag_enable_social_icons', 0 );
			}

			// Disable social icon on header if enabled on footer only.
			if ( 'footer' === $social_icon_visibility ) {
				set_theme_mod( 'colormag_social_icons_header_activate', false );
			}

			// Disable social icon on footer if enabled on header only.
			if ( 'header' === $social_icon_visibility ) {
				set_theme_mod( 'colormag_social_icons_footer_activate', false );
			}

			$remove_theme_mod_settings = array(
				'colormag_social_link_activate',
				'colormag_social_link_location_option',
			);

			// Loop through the theme mods to remove them.
			foreach ( $remove_theme_mod_settings as $remove_theme_mod_setting ) {
				remove_theme_mod( $remove_theme_mod_setting );
			}

			// Set flag to not repeat the migration process, ie, run it only once.
			update_option( 'colormag_social_icons_control_migrate', true );
		}

		public function colormag_post_meta_control_migrate() {

			// Set flag to not repeat the migration process, ie, run it only once.
			update_option( 'colormag_post_meta_control_migrate', true );

			$demo_import_migration = self::colormag_demo_import_migration();

			if ( ! $demo_import_migration ) {
				if ( get_option( 'colormag_post_meta_control_migrate', false ) ) {
					return;
				}
			}

			// Associative array that holds new mod setting as a key and prev mod setting as value.
			$prev_mod['categories'] = get_theme_mod( 'colormag_category_entry_meta_remove', 0 );
			$prev_mod['date']       = get_theme_mod( 'colormag_date_entry_meta_remove', 0 );
			$prev_mod['author']     = get_theme_mod( 'colormag_author_entry_meta_remove', 0 );
			$prev_mod['views']      = get_theme_mod( 'colormag_post_view_entry_meta_remove', 0 );
			$prev_mod['comments']   = get_theme_mod( 'colormag_comments_entry_meta_remove', 0 );
			$prev_mod['tags']       = get_theme_mod( 'colormag_tags_entry_meta_remove', 0 );
			$prev_mod['read-time']  = get_theme_mod( 'colormag_reading_time_setting', 0 );

			// Will contain data that would be shown in the front-end.
			$new_mod = array();

			// Add values to $new_mod array.
			foreach ( $prev_mod as $key => $value ) {
				if ( 0 === $value && 'read-time' !== $key ) {
					array_push( $new_mod, $key );
				}
			}

			// As read time is shown in the front-end when value is 1.
			if ( 0 !== $prev_mod['read-time'] ) {
				array_push( $new_mod, 'read-time' );
			}

			set_theme_mod( 'colormag_post_meta_structure', $new_mod );

			// Remove prev theme mods.
			$prev_theme_mod_settings = array(
				'colormag_category_entry_meta_remove',
				'colormag_date_entry_meta_remove',
				'colormag_author_entry_meta_remove',
				'colormag_post_view_entry_meta_remove',
				'colormag_comments_entry_meta_remove',
				'colormag_tags_entry_meta_remove',
				'colormag_reading_time_setting',
			);

			// Loop through prev theme mods to remove them.
			foreach ( $prev_theme_mod_settings as $prev_theme_mod_setting ) {
				remove_theme_mod( $prev_theme_mod_setting );
			}

			// Set flag to not repeat the migration process, ie, run it only once.
			update_option( 'colormag_post_meta_control_migrate', true );
		}

		public function colormag_menu_logo_icon_control_migrate() {

			$demo_import_migration = self::colormag_demo_import_migration();

			// Migrate the customize option if migration is done manually.
			if ( ! $demo_import_migration ) {
				// Bail out if the migration is already done.
				if ( get_option( 'colormag_menu_logo_icon_control_migrate' ) ) {
					return;
				}
			}

			// Assign the required variables.
			$home_icon_visibility = get_theme_mod( 'colormag_menu_icon_logo', 'none' );

			/**
			 * Update the theme mods data.
			 */
			// Set option to home icon if enabled.
			if ( 'home_icon' == $home_icon_visibility ) {
				set_theme_mod( 'colormag_menu_icon_logo', 'home-icon' );
			} elseif ( 'logo' == $home_icon_visibility ) {
				set_theme_mod( 'colormag_menu_icon_logo', 'logo' );
			}

			// Set option to none if home icon disabled.
			if ( 0 == $home_icon_visibility ) {
				set_theme_mod( 'colormag_menu_icon_logo', 'none' );
			}

			// Remove unnecessary theme mods data.
			$remove_theme_mod_settings = array(
				'colormag_home_icon_display',
			);

			// Loop through the theme mods to remove them.
			foreach ( $remove_theme_mod_settings as $remove_theme_mod_setting ) {
				remove_theme_mod( $remove_theme_mod_setting );
			}

			// Set flag to not repeat the migration process, ie, run it only once.
			update_option( 'colormag_menu_logo_icon_control_migrate', true );

			// Set flag for demo import migration to not repeat the migration process, ie, run it only once.
			if ( $demo_import_migration ) {
				update_option( 'colormag_demo_import_migration_notice_dismiss', true );
			}
		}

		public function colormag_top_bar_options_migrate() {

			$demo_import_migration = self::colormag_demo_import_migration();

			// Migrate the customize option if migration is done manually.
			if ( ! $demo_import_migration ) {
				// Bail out if the migration is already done.
				if ( get_option( 'colormag_top_bar_options_migrate' ) ) {
					return;
				}
			}

			$top_bar_date_display          = get_theme_mod( 'colormag_date_display', 0 );
			$top_bar_breaking_news_display = get_theme_mod( 'colormag_enable_news_ticker', 0 );
			$top_bar_social_icons_display  = get_theme_mod( 'colormag_enable_social_icons', 0 );
			$top_bar_social_icons_location = get_theme_mod( 'colormag_social_icons_header_location', 'top-bar' );
			$top_bar_menu_display          = get_theme_mod( 'colormag_top_bar_menu_enable', 0 );

			// Enable top bar if top bar date is enabled.
			if ( 1 == $top_bar_date_display ) {
				set_theme_mod( 'colormag_enable_top_bar', '1' );
			}

			// Enable top bar if top bar breaking news is enabled.
			if ( 1 == $top_bar_breaking_news_display ) {
				set_theme_mod( 'colormag_enable_top_bar', '1' );
			}

			// Enable top bar if top bar social icon is enabled and social icon location is set to top bar.
			if ( 1 == $top_bar_social_icons_display && 'top-bar' == $top_bar_social_icons_location ) {
				set_theme_mod( 'colormag_enable_top_bar', '1' );
			}

			// Enable top bar if top bar menu is enabled.
			if ( 1 == $top_bar_menu_display ) {
				set_theme_mod( 'colormag_enable_top_bar', '1' );
			}

			// Set flag to not repeat the migration process, ie, run it only once.
			update_option( 'colormag_top_bar_options_migrate', true );

			// Set flag for demo import migration to not repeat the migration process, ie, run it only once.
			if ( $demo_import_migration ) {
				update_option( 'colormag_demo_import_migration_notice_dismiss', true );
			}
		}

		public function colormag_breadcrumb_options_migrate() {

			$demo_import_migration = self::colormag_demo_import_migration();

			// Migrate the customize option if migration is done manually.
			if ( ! $demo_import_migration ) {

				// Bail out if the migration is already done.
				if ( get_option( 'colormag_breadcrumb_options_migrate' ) ) {
					return;
				}
			}

			$breadcrumb_display = get_theme_mod( 'colormag_breadcrumb_display' );

			if ( 1 === $breadcrumb_display ) {
				set_theme_mod( 'colormag_breadcrumb_display', 'yoast_seo_navxt' );
			} else {
				set_theme_mod( 'colormag_breadcrumb_display', 'none' );
			}

			// Set flag to not repeat the migration process, ie, run it only once.
			update_option( 'colormag_breadcrumb_options_migrate', true );

			// Set flag for demo import migration to not repeat the migration process, ie, run it only once.
			if ( $demo_import_migration ) {
				update_option( 'colormag_demo_import_migration_notice_dismiss', true );
			}
		}

		public function colormag_autoload_posts_control_migrate() {

			// Bail out if the migration is already done.
			if ( get_option( 'colormag_autoload_posts_control_migrate' ) ) {
				return;
			}

			$autoload_posts       = get_theme_mod( 'colormag_load_next_post', 0 );
			$autoload_posts_limit = get_theme_mod( 'colormag_load_next_post_limit', 2 );
			$autoload_posts_event = get_theme_mod( 'colormag_load_next_post_event', 'button' );

			if ( 0 !== $autoload_posts ) {
				set_theme_mod( 'colormag_enable_autoload_posts', $autoload_posts );
			}

			if ( 2 !== $autoload_posts_limit ) {
				set_theme_mod( 'colormag_autoload_posts_limit', $autoload_posts_limit );
			}

			if ( 'button' !== $autoload_posts_event ) {
				set_theme_mod( 'colormag_autoload_posts_event', $autoload_posts_event );
			}

			// Remove old theme mods.
			remove_theme_mod( 'colormag_load_next_post' );
			remove_theme_mod( 'colormag_load_next_post_limit' );
			remove_theme_mod( 'colormag_load_next_post_event' );

			// Set flag to not repeat the migration process, ie, run it only once.
			update_option( 'colormag_autoload_posts_control_migrate', true );
		}

		public function colormag_major_update_migration_v3() {

			/**
			 * Color control migration.
			 */
			$color_options = array(
				array(
					'old_key' => 'colormag_breaking_news_content_option',
					'new_key' => 'colormag_news_ticker_label',
				),
				array(
					'old_key' => 'colormag_read_more_heading',
					'new_key' => 'colormag_read_more_text',
				),
				array(
					'old_key' => 'colormag_content_text_color',
					'new_key' => 'colormag_base_color',
				),
				array(
					'old_key' => 'colormag_content_part_title_color',
					'new_key' => 'colormag_headings_color',
				),
				array(
					'old_key' => 'colormag_button_text_color',
					'new_key' => 'colormag_button_color',
				),
				array(
					'old_key' => 'colormag_breaking_news_label_background_color',
					'new_key' => 'colormag_news_ticker_label_background_color',
				),
				array(
					'old_key' => 'colormag_progress_bar_bgcolor',
					'new_key' => 'colormag_progress_bar_indicator_color',
				),
				array(
					'old_key' => 'colormag_footer_small_menu_text_color',
					'new_key' => 'colormag_footer_menu_color',
				),
				array(
					'old_key' => 'colormag_footer_small_menu_text_hover_color',
					'new_key' => 'colormag_footer_menu_hover_color',
				),
			);

			foreach ( $color_options as $option ) {
				$old_value = get_theme_mod( $option['old_key'] );

				if ( $old_value ) {
					set_theme_mod( $option['new_key'], $old_value );

					remove_theme_mod( $option['old_key'] );
				}
			}

			/**
			 * Select control migration.
			 */
			$breadcrumb_option = get_theme_mod( 'colormag_breadcrumb_display', 'colormag_breadcrumb' );

			if ( 'yoast_seo_navxt' == $breadcrumb_option || 'colormag_breadcrumb' == $breadcrumb_option ) {
				set_theme_mod( 'colormag_breadcrumb_enable', 1 );
				set_theme_mod( 'colormag_breadcrumb_type', $breadcrumb_option );
				remove_theme_mod( 'colormag_breadcrumb_display' );
			}

			$container_layout = get_theme_mod( 'colormag_site_layout' );

			if ( $container_layout ) {
				$new_container_layout = '';
				if ( 'boxed_layout' === $container_layout ) {
					$new_container_layout = 'boxed';
				} elseif ( 'wide_layout' === $container_layout ) {
					$new_container_layout = 'wide';
				}
				set_theme_mod( 'colormag_container_layout', $new_container_layout );
				remove_theme_mod( 'colormag_site_layout' );

			}

			// Site identity placement.
			$header_logo_placement = get_theme_mod( 'colormag_header_logo_placement', 'header_text_only' );

			switch ( $header_logo_placement ) {
				case 'disable':
				case 'header_logo_only':
					set_theme_mod( 'colormag_enable_site_identity', 0 );
					set_theme_mod( 'colormag_enable_site_tagline', 0 );
					break;
				case 'show_both':
				case 'header_text_only':
					set_theme_mod( 'colormag_enable_site_identity', 1 );
					set_theme_mod( 'colormag_enable_site_tagline', 1 );
					break;
			}
			remove_theme_mod( 'colormag_site_layout' );

			// Header media position.
			$old_header_media_position = get_theme_mod( 'colormag_header_image_position' );

			if ( $old_header_media_position ) {

				if ( 'position_one' === $old_header_media_position ) {
					$new_header_media_position = 'position-one';
				} elseif ( 'position_two' === $old_header_media_position ) {
					$new_header_media_position = 'position-two';
				} elseif ( 'position_three' === $old_header_media_position ) {
					$new_header_media_position = 'position-three';
				} else {
					$new_header_media_position = 'position-one';
				}

				set_theme_mod( 'colormag_header_media_position', $new_header_media_position );
				remove_theme_mod( 'colormag_header_image_position' );
			}

			// Main Header layout.
			$old_main_header_layout = get_theme_mod( 'colormag_main_total_header_area_display_type' );

			if ( $old_main_header_layout ) {

				if ( 'type_one' === $old_main_header_layout ) {
					set_theme_mod( 'colormag_main_header_layout', 'layout-1' );
					set_theme_mod( 'colormag_main_header_layout_1_style', 'style-1' );
				} elseif ( 'type_two' === $old_main_header_layout ) {
					set_theme_mod( 'colormag_main_header_layout', 'layout-1' );
					set_theme_mod( 'colormag_main_header_layout_1_style', 'style-1' );
					set_theme_mod( 'colormag_main_header_width_setting', 'contained' );
				} elseif ( 'type_three' === $old_main_header_layout ) {
					set_theme_mod( 'colormag_main_header_layout', 'layout-2' );
					set_theme_mod( 'colormag_main_header_layout_2_style', 'style-1' );
				} elseif ( 'type_four' === $old_main_header_layout ) {
					set_theme_mod( 'colormag_main_header_layout', 'layout-1' );
					set_theme_mod( 'colormag_main_header_layout_1_style', 'style-2' );
				} elseif ( 'type_five' === $old_main_header_layout ) {
					set_theme_mod( 'colormag_main_header_layout', 'layout-3' );
					set_theme_mod( 'colormag_main_header_layout_3_style', 'style-1' );
				} elseif ( 'type_six' === $old_main_header_layout ) {
					set_theme_mod( 'colormag_main_header_layout', 'layout-2' );
					set_theme_mod( 'colormag_main_header_layout_2_style', 'style-2' );
				} else {
					set_theme_mod( 'colormag_main_header_layout', 'layout-1' );
					set_theme_mod( 'colormag_main_header_layout_1_style', 'style-1' );
				}
				remove_theme_mod( 'colormag_main_total_header_area_display_type' );
			}

			// Header media position.
			$old_menu_icon_option = get_theme_mod( 'colormag_menu_icon_logo' );

			if ( $old_menu_icon_option ) {

				if ( 'home_icon' === $old_menu_icon_option ) {
					$new_menu_icon_option = 'home-icon';
				} elseif ( 'logo' === $old_menu_icon_option ) {
					$new_menu_icon_option = 'logo';
				} elseif ( 'none' === $old_menu_icon_option ) {
					$new_menu_icon_option = 'none';
				} else {
					$new_menu_icon_option = 'none';
				}

				set_theme_mod( 'colormag_menu_icon_logo', $new_menu_icon_option );
			}

			// News Ticker position.
			$old_news_ticker_position = get_theme_mod( 'colormag_breaking_news_position_options' );

			if ( $old_news_ticker_position ) {

				if ( 'header' === $old_news_ticker_position ) {
					$new_news_ticker_position = 'top-bar';
				} elseif ( 'main' === $old_news_ticker_position ) {
					$new_news_ticker_position = 'below-header';
				} else {
					$new_news_ticker_position = 'top-bar';
				}

				set_theme_mod( 'colormag_news_ticker_position', $new_news_ticker_position );
				remove_theme_mod( 'colormag_breaking_news_position_options' );
			}

			// News Ticker query.
			$old_news_ticker_query = get_theme_mod( 'colormag_breaking_news_category_option' );

			if ( $old_news_ticker_query ) {

				if ( 'latest' === $old_news_ticker_query ) {
					$new_news_ticker_query = 'latest';
				} elseif ( 'category' === $old_news_ticker_query ) {
					$new_news_ticker_query = 'category';
				} else {
					$new_news_ticker_query = 'latest';
				}

				set_theme_mod( 'colormag_news_ticker_query', $new_news_ticker_query );
				remove_theme_mod( 'colormag_breaking_news_category_option' );
			}

			// News Ticker animation.
			$old_news_ticker_animation = get_theme_mod( 'colormag_breaking_news_setting_animation_options' );

			if ( $old_news_ticker_animation ) {

				if ( 'up' === $old_news_ticker_animation ) {
					$new_news_ticker_animation = 'up';
				} elseif ( 'down' === $old_news_ticker_animation ) {
					$new_news_ticker_animation = 'down';
				} else {
					$new_news_ticker_animation = 'down';
				}

				set_theme_mod( 'colormag_news_ticker_animation_direction', $new_news_ticker_animation );
				remove_theme_mod( 'colormag_breaking_news_setting_animation_options' );
			}

			// Blog layout.
			$old_blog_layout = get_theme_mod( 'colormag_archive_search_layout' );

			if ( $old_blog_layout ) {

				if ( 'double_column_layout' === $old_blog_layout ) {
					set_theme_mod( 'colormag_blog_layout', 'layout-2' );
					set_theme_mod( 'colormag_blog_layout_1_style', 'style-1' );
				} elseif ( 'single_column_layout' === $old_blog_layout ) {
					set_theme_mod( 'colormag_blog_layout', 'layout-1' );
					set_theme_mod( 'colormag_blog_layout_1_style', 'style-1' );
				} elseif ( 'full_width_layout' === $old_blog_layout ) {
					set_theme_mod( 'colormag_blog_layout', 'layout-1' );
					set_theme_mod( 'colormag_blog_layout_1_style', 'style-2' );
				} elseif ( 'grid_layout' === $old_blog_layout ) {
					set_theme_mod( 'colormag_blog_layout', 'layout-2' );
					set_theme_mod( 'colormag_blog_layout_1_style', 'style-2' );
				} else {
					set_theme_mod( 'colormag_blog_layout', 'layout-1' );
					set_theme_mod( 'colormag_blog_layout_1_style', 'style-1' );
				}

				remove_theme_mod( 'colormag_archive_search_layout' );
			}

			// Grid layout column.
			$old_grid_layout_column = get_theme_mod( 'colormag_grid_layout_column' );

			if ( $old_grid_layout_column ) {

				if ( 'two' === $old_grid_layout_column ) {
					$new_grid_layout_column = '2';
				} elseif ( 'three' === $old_grid_layout_column ) {
					$new_grid_layout_column = '3';
				} elseif ( 'four' === $old_grid_layout_column ) {
					$new_grid_layout_column = '4';
				} else {
					$new_grid_layout_column = '2';
				}

				set_theme_mod( 'colormag_grid_layout_column', $new_grid_layout_column );
			}

			// Footer bar layout.
			$old_footer_bar_layout = get_theme_mod( 'colormag_footer_copyright_alignment_setting' );

			if ( $old_footer_bar_layout ) {

				if ( 'left' === $old_footer_bar_layout ) {
					$new_footer_bar_layout = 'left';
				} elseif ( 'right' === $old_footer_bar_layout ) {
					$new_footer_bar_layout = 'right';
				} elseif ( 'center' === $old_footer_bar_layout ) {
					$new_footer_bar_layout = 'center';
				} else {
					$new_footer_bar_layout = 'left';
				}

				set_theme_mod( 'colormag_footer_bar_alignment', $new_footer_bar_layout );
				remove_theme_mod( 'colormag_footer_copyright_alignment_setting' );
			}

			// Main footer layout.
			$old_main_footer_layout = get_theme_mod( 'colormag_main_footer_layout_display_type' );

			if ( $old_main_footer_layout ) {

				if ( 'type_one' === $old_main_footer_layout ) {
					$new_main_footer_layout = 'layout-1';
				} elseif ( 'type_two' === $old_main_footer_layout ) {
					$new_main_footer_layout = 'layout-2';
				} elseif ( 'type_three' === $old_main_footer_layout ) {
					$new_main_footer_layout = 'layout-3';
				} else {
					$new_main_footer_layout = 'layout-1';
				}

				set_theme_mod( 'colormag_main_footer_layout', $new_main_footer_layout );
				remove_theme_mod( 'colormag_main_footer_layout_display_type' );
			}

			// Post Meta date style.
			$old_post_meta_date_style = get_theme_mod( 'colormag_post_meta_date_setting' );

			if ( $old_post_meta_date_style ) {

				if ( 'post_date' === $old_post_meta_date_style ) {
					$new_post_meta_date_style = 'style-1';
				} elseif ( 'post_human_readable_date' === $old_post_meta_date_style ) {
					$new_post_meta_date_style = 'style-2';
				} else {
					$new_post_meta_date_style = 'style-1';
				}

				set_theme_mod( 'colormag_post_meta_date_style', $new_post_meta_date_style );
				remove_theme_mod( 'colormag_post_meta_date_setting' );
			}

			// Related post flyout query.
			$old_related_post_flyout_query = get_theme_mod( 'colormag_related_posts_flyout_type' );

			if ( $old_related_post_flyout_query ) {

				if ( 'categories' === $old_related_post_flyout_query ) {
					$new_related_post_flyout_query = 'categories';
				} elseif ( 'tags' === $old_related_post_flyout_query ) {
					$new_related_post_flyout_query = 'tags';
				} elseif ( 'date' === $old_related_post_flyout_query ) {
					$new_related_post_flyout_query = 'date';
				} else {
					$new_related_post_flyout_query = 'categories';
				}

				set_theme_mod( 'colormag_flyout_related_posts_query', $new_related_post_flyout_query );
				remove_theme_mod( 'colormag_related_posts_flyout_type' );
			}

			// Related post layout.
			$old_related_posts_layout = get_theme_mod( 'colormag_related_posts_layout' );

			if ( $old_related_posts_layout ) {

				if ( 'style_one' === $old_related_posts_layout ) {
					$new_related_posts_layout = 'style-1';
				} elseif ( 'style_two' === $old_related_posts_layout ) {
					$new_related_posts_layout = 'style-2';
				} elseif ( 'style_three' === $old_related_posts_layout ) {
					$new_related_posts_layout = 'style-3';
				} elseif ( 'style_four' === $old_related_posts_layout ) {
					$new_related_posts_layout = 'style-4';
				} else {
					$new_related_posts_layout = 'style-1';
				}

				set_theme_mod( 'colormag_related_posts_style', $new_related_posts_layout );
				remove_theme_mod( 'colormag_related_posts_flyout_type' );
			}

			// Related post count.
			$old_related_posts_count = get_theme_mod( 'colormag_related_post_number_display' );

			if ( $old_related_posts_count ) {

				if ( '3' === $old_related_posts_count ) {
					$new_related_posts_count = '3';
				} elseif ( '6' === $old_related_posts_count ) {
					$new_related_posts_count = '6';
				} else {
					$new_related_posts_count = 'style-1';
				}

				set_theme_mod( 'colormag_related_posts_count', $new_related_posts_count );
				remove_theme_mod( 'colormag_related_post_number_display' );
			}

			// Related post flyout query.
			$old_related_post_query = get_theme_mod( 'colormag_related_posts' );

			if ( $old_related_post_query ) {

				if ( 'categories' === $old_related_post_query ) {
					$new_related_post_query = 'categories';
				} elseif ( 'tags' === $old_related_post_query ) {
					$new_related_post_query = 'tags';
				} else {
					$new_related_post_query = 'categories';
				}

				set_theme_mod( 'colormag_related_posts_query', $new_related_post_query );
				remove_theme_mod( 'colormag_related_posts' );
			}

			// Post Navigation.
			$old_post_navigation_style = get_theme_mod( 'colormag_post_navigation' );

			if ( $old_post_navigation_style ) {

				if ( 'default' === $old_post_navigation_style ) {
					$new_post_navigation_style = 'style-1';
				} elseif ( 'small_featured_image' === $old_post_navigation_style ) {
					$new_post_navigation_style = 'style-2';
				} elseif ( 'medium_featured_image' === $old_post_navigation_style ) {
					$new_post_navigation_style = 'style-3';
				} else {
					$new_post_navigation_style = 'style-1';
				}

				set_theme_mod( 'colormag_post_navigation_style', $new_post_navigation_style );
				remove_theme_mod( 'colormag_post_navigation' );
			}

			// Single post title.
			$old_single_post_title_position = get_theme_mod( 'colormag_single_post_title_position' );

			if ( $old_single_post_title_position ) {

				if ( 'above' === $old_single_post_title_position ) {
					$new_single_post_title_position = 'position-1';
				} elseif ( 'below' === $old_single_post_title_position ) {
					$new_single_post_title_position = 'position-2';
				} else {
					$new_single_post_title_position = 'position-2';
				}

				set_theme_mod( 'colormag_featured_image_position', $new_single_post_title_position );
				remove_theme_mod( 'colormag_single_post_title_position' );
			}

			// Single post title.
			$old_single_post_title_position = get_theme_mod( 'colormag_author_bio_style_setting' );

			if ( $old_single_post_title_position ) {

				if ( 'style_one' === $old_single_post_title_position ) {
					$new_single_post_title_position = 'style-1';
				} elseif ( 'style_two' === $old_single_post_title_position ) {
					$new_single_post_title_position = 'style-2';
				} elseif ( 'style_three' === $old_single_post_title_position ) {
					$new_single_post_title_position = 'style-3';
				} else {
					$new_single_post_title_position = 'style-1';
				}

				set_theme_mod( 'colormag_author_bio_style', $new_single_post_title_position );
				remove_theme_mod( 'colormag_author_bio_style_setting' );
			}

			// Autoload post type.
			$old_autoload_posts_type = get_theme_mod( 'colormag_autoload_posts_event' );

			if ( $old_autoload_posts_type ) {

				if ( 'button' === $old_autoload_posts_type ) {
					$new_autoload_posts_type = 'button';
				} elseif ( 'scroll' === $old_autoload_posts_type ) {
					$new_autoload_posts_type = 'scroll';
				} else {
					$new_autoload_posts_type = 'button';
				}

				set_theme_mod( 'colormag_autoload_posts_type', $new_autoload_posts_type );
				remove_theme_mod( 'colormag_autoload_posts_event' );
			}

			// Breadcrumb.
			$old_breadcrumb = get_theme_mod( 'colormag_breadcrumb_display' );

			if ( $old_breadcrumb ) {
				if ( 'none' === $old_breadcrumb ) {
					set_theme_mod( 'colormag_enable_breadcrumb', false );
				} elseif ( 'yoast_seo_navxt' === $old_breadcrumb ) {
					set_theme_mod( 'colormag_enable_breadcrumb', true );
				} elseif ( 'colormag_breadcrumb' === $old_breadcrumb ) {
					set_theme_mod( 'colormag_enable_breadcrumb', true );
				}
			}

			// Breadcrumb type.
			$old_breadcrumb_type = get_theme_mod( 'colormag_breadcrumb_display' );

			if ( $old_breadcrumb_type ) {
				if ( 'none' === $old_breadcrumb_type ) {
					$new_breadcrumb_type = 'colormag_breadcrumb';
				} elseif ( 'yoast_seo_navxt' === $old_breadcrumb_type ) {
					$new_breadcrumb_type = 'yoast_seo_navxt';
				} elseif ( 'colormag_breadcrumb' === $old_breadcrumb_type ) {
					$new_breadcrumb_type = 'colormag_breadcrumb';
				} else {
					$new_breadcrumb_type = 'button';
				}

				set_theme_mod( 'colormag_breadcrumb_type', $new_breadcrumb_type );
				remove_theme_mod( 'colormag_breadcrumb_display' );
			}

			// Sticky menu type.
			$old_sticky_menu_style = get_theme_mod( 'colormag_primary_sticky_menu_type' );

			if ( $old_sticky_menu_style ) {
				if ( 'sticky' === $old_sticky_menu_style ) {
					$new_sticky_menu_style = 'sticky';
				} elseif ( 'reveal_on_scroll' === $old_sticky_menu_style ) {
					$new_sticky_menu_style = 'reveal_on_scroll';
				}

				set_theme_mod( 'colormag_sticky_menu_type', $new_sticky_menu_style );
				remove_theme_mod( 'colormag_primary_sticky_menu_type' );
			}

			// Pagination enable.
			$old_pagination = get_theme_mod( 'colormag_post_pagination' );

			if ( $old_pagination ) {
				set_theme_mod( 'colormag_enable_pagination', 1 );
			}

			// Blog content excerpt type.
			$old_blog_content_excerpt_type = get_theme_mod( 'colormag_archive_content_excerpt_display' );

			if ( $old_blog_content_excerpt_type ) {
				if ( 'excerpt' === $old_blog_content_excerpt_type ) {
					$new_blog_content_excerpt_type = 'excerpt';
				} elseif ( 'content' === $old_blog_content_excerpt_type ) {
					$new_blog_content_excerpt_type = 'content';
				} else {
					$new_blog_content_excerpt_type = 'excerpt';
				}

				set_theme_mod( 'colormag_blog_content_excerpt_type', $new_blog_content_excerpt_type );
				remove_theme_mod( 'colormag_archive_content_excerpt_display' );
			}

			// Pagination type.
			$old_pagination_type = get_theme_mod( 'colormag_post_pagination' );

			if ( $old_pagination_type ) {
				if ( 'default' === $old_pagination_type ) {
					$new_pagination_type = 'default';
				} elseif ( 'numbered_pagination' === $old_pagination_type ) {
					$new_pagination_type = 'numbered_pagination';
				} elseif ( 'infinite_scroll' === $old_pagination_type ) {
					$new_pagination_type = 'infinite_scroll';
				} else {
					$new_pagination_type = 'default';
				}

				set_theme_mod( 'colormag_pagination_type', $new_pagination_type );
				remove_theme_mod( 'colormag_post_pagination' );
			}

			// Infinite scroll type.
			$old_pagination_type = get_theme_mod( 'colormag_infinite_scroll_event' );

			if ( $old_pagination_type ) {
				if ( 'button' === $old_pagination_type ) {
					$new_pagination_type = 'button';
				} elseif ( 'scroll' === $old_pagination_type ) {
					$new_pagination_type = 'scroll';
				} else {
					$new_pagination_type = 'button';
				}

				set_theme_mod( 'colormag_infinite_scroll_type', $new_pagination_type );
				remove_theme_mod( 'colormag_infinite_scroll_event' );
			}

			/**
			 * Toggle control migration.
			 */
			// Sticky sidebar.
			$enable_sticky_sidebar = get_theme_mod( 'colormag_sticky_content_sidebar' );

			if ( $enable_sticky_sidebar ) {
				set_theme_mod( 'colormag_enable_sticky_sidebar', true );
			}
			remove_theme_mod( 'colormag_sticky_content_sidebar' );

			// Hide Blog/Static page posts.
			$blog_static_page_posts = get_theme_mod( 'colormag_hide_blog_front' );

			if ( $blog_static_page_posts ) {
				set_theme_mod( 'colormag_hide_blog_static_page_post', true );
			}
			remove_theme_mod( 'colormag_hide_blog_front' );

			// Unique Post System.
			$enable_unique_post_system = get_theme_mod( 'colormag_unique_post_system' );

			if ( $enable_unique_post_system ) {
				set_theme_mod( 'colormag_enable_unique_post_system', true );
			}
			remove_theme_mod( 'colormag_unique_post_system' );

			// Unique Post System.
			$enable_header_image_link_home = get_theme_mod( 'colormag_header_image_link' );

			if ( $enable_header_image_link_home ) {
				set_theme_mod( 'colormag_enable_header_image_link_home', true );
			}
			remove_theme_mod( 'colormag_header_image_link' );

			// Enable News Ticker.
			$enable_news_ticker = get_theme_mod( 'colormag_breaking_news' );

			if ( $enable_news_ticker ) {
				set_theme_mod( 'colormag_enable_news_ticker', true );
			}
			remove_theme_mod( 'colormag_breaking_news' );

			// Footer Social Icons.
			$enable_social_icon = get_theme_mod( 'colormag_social_icons_activate' );

			if ( ! empty( $enable_social_icon ) ) {
				set_theme_mod( 'colormag_enable_social_icons', 1 );
			}
			remove_theme_mod( 'colormag_social_icons_activate' );

			// Header Social Icons.
			$enable_header_social_icons = get_theme_mod( 'colormag_social_icons_header_activate' );

			if ( ! empty( $enable_header_social_icons ) ) {
				set_theme_mod( 'colormag_enable_social_icons_header', 1 );
			}
			remove_theme_mod( 'colormag_social_icons_header_activate' );

			// Social Icons.
			$enable_social_icons = get_theme_mod( 'colormag_social_icons_activate' );

			if ( ! empty( $enable_social_icons ) ) {
				set_theme_mod( 'colormag_enable_social_icon', 1 );
			}
			remove_theme_mod( 'colormag_social_icons_activate' );

			// Schema Markup.
			$enable_schema_markup = get_theme_mod( 'colormag_enable_schema_markup' );

			if ( $enable_schema_markup ) {
				set_theme_mod( 'colormag_enable_schema_markup', true );
			}
			remove_theme_mod( 'colormag_schema_markup' );

			// Scroll to top.
			$enable_scroll_to_top = get_theme_mod( 'colormag_enable_scroll_to_top' );

			if ( $enable_scroll_to_top ) {
				set_theme_mod( 'colormag_enable_scroll_to_top', true );
			}
			remove_theme_mod( 'colormag_scroll_to_top' );

			// Post Featured Image.
			$enable_post_featured_image = get_theme_mod( 'colormag_featured_image_show' );

			if ( ! empty( $enable_post_featured_image ) ) {
				set_theme_mod( 'colormag_enable_featured_image', false );
			} else {
				set_theme_mod( 'colormag_enable_featured_image', true );
			}

			// Page Featured Image.
			$enable_page_featured_image = get_theme_mod( 'colormag_featured_image_single_page_show' );

			if ( $enable_page_featured_image ) {
				set_theme_mod( 'colormag_enable_page_featured_image', true );
			}
			remove_theme_mod( 'colormag_featured_image_single_page_show' );

			// Progress bar indicator.
			$enable_progress_bar_indicator = get_theme_mod( 'colormag_prognroll_indicator' );

			if ( $enable_progress_bar_indicator ) {
				set_theme_mod( 'colormag_enable_progress_bar_indicator', true );
			}
			remove_theme_mod( 'colormag_prognroll_indicator' );

			// Flyout related posts.
			$enable_fly_out_related_post = get_theme_mod( 'colormag_related_post_flyout_setting' );

			if ( $enable_fly_out_related_post ) {
				set_theme_mod( 'colormag_enable_flyout_related_posts', true );
			}
			remove_theme_mod( 'colormag_related_post_flyout_setting' );

			// Related posts.
			$enable_related_post = get_theme_mod( 'colormag_related_posts_activate' );

			if ( $enable_related_post ) {
				set_theme_mod( 'colormag_enable_related_posts', true );
			}
			remove_theme_mod( 'colormag_related_posts_activate' );

			// Post Navigation.
			$enable_post_navigation = get_theme_mod( 'colormag_post_navigation_hide' );

			if ( $enable_post_navigation ) {
				set_theme_mod( 'colormag_enable_post_navigation', true );
			}
			remove_theme_mod( 'colormag_post_navigation_hide' );

			// Lightbox.
			$enable_lightbox = get_theme_mod( 'colormag_featured_image_popup' );

			if ( $enable_lightbox ) {
				set_theme_mod( 'colormag_enable_lightbox', true );
			}
			remove_theme_mod( 'colormag_featured_image_popup' );

			// Social Share.
			$enable_social_share = get_theme_mod( 'colormag_social_share' );

			if ( $enable_social_share ) {
				set_theme_mod( 'colormag_enable_social_share', true );
			}
			remove_theme_mod( 'colormag_social_share' );

			// Social Share Twitter.
			$enable_social_share_twitter = get_theme_mod( 'colormag_social_share_twitter' );

			if ( $enable_social_share_twitter ) {
				set_theme_mod( 'colormag_enable_social_share_twitter', true );
			}
			remove_theme_mod( 'colormag_social_share_twitter' );

			// Social Share Facebook.
			$enable_social_share_facebook = get_theme_mod( 'colormag_social_share_facebook' );

			if ( $enable_social_share_facebook ) {
				set_theme_mod( 'colormag_enable_social_share_facebook', true );
			}
			remove_theme_mod( 'colormag_social_share_facebook' );

			// Social Share Pinterest.
			$enable_social_share_pinterest = get_theme_mod( 'colormag_social_share_pinterest' );

			if ( $enable_social_share_pinterest ) {
				set_theme_mod( 'colormag_enable_social_share_pinterest', true );
			}
			remove_theme_mod( 'colormag_social_share_pinterest' );

			// Author bio link.
			$enable_author_bio_link = get_theme_mod( 'colormag_author_bio_links' );

			if ( ! empty( $enable_author_bio_link ) ) {
				set_theme_mod( 'colormag_enable_author_bio', false );
			} else {
				set_theme_mod( 'colormag_enable_author_bio', true );
			}

			// Author bio profile.
			$enable_author_bio_profile = get_theme_mod( 'colormag_author_bio_social_sites_show' );

			if ( $enable_author_bio_profile ) {
				set_theme_mod( 'colormag_enable_author_bio_profile', true );
			}
			remove_theme_mod( 'colormag_author_bio_social_sites_show' );

			// Author bio.
			$enable_author_bio = get_theme_mod( 'colormag_author_bio_disable_setting' );

			if ( $enable_author_bio ) {
				set_theme_mod( 'colormag_enable_author_bio', false );
			}
			remove_theme_mod( 'colormag_author_bio_disable_setting' );

			// Autoload posts.
			$enable_autoload_posts = get_theme_mod( 'colormag_autoload_posts' );

			if ( $enable_autoload_posts ) {
				set_theme_mod( 'colormag_enable_autoload_posts', true );
			}
			remove_theme_mod( 'colormag_autoload_posts' );

			// Sticky Menu.
			$enable_sticky_menu = get_theme_mod( 'colormag_primary_sticky_menu' );

			if ( $enable_sticky_menu ) {
				set_theme_mod( 'colormag_enable_sticky_menu', true );
			}
			remove_theme_mod( 'colormag_primary_sticky_menu' );

			// Search in menu.
			$enable_search_in_menu = get_theme_mod( 'colormag_search_icon_in_menu' );

			if ( $enable_search_in_menu ) {
				set_theme_mod( 'colormag_enable_search', true );
			}
			remove_theme_mod( 'colormag_search_icon_in_menu' );

			// Random post.
			$enable_random_post = get_theme_mod( 'colormag_random_post_in_menu' );

			if ( $enable_random_post ) {
				set_theme_mod( 'colormag_enable_random_post', true );
			}
			remove_theme_mod( 'colormag_random_post_in_menu' );

			// Category color.
			$enable_category_color = get_theme_mod( 'colormag_category_menu_color' );

			if ( $enable_category_color ) {
				set_theme_mod( 'colormag_enable_category_color', true );
			}
			remove_theme_mod( 'colormag_category_menu_color' );

			// Featured image caption.
			$enable_category_color = get_theme_mod( 'colormag_featured_image_caption_show' );

			if ( $enable_category_color ) {
				set_theme_mod( 'colormag_enable_featured_image_caption', true );
			}
			remove_theme_mod( 'colormag_featured_image_caption_show' );

			/**
			 * Typography control migration.
			 */
			// Migrate the typography options.
			$extractsizeandunit = function ( $value ) {
				$unit_list = array( 'px', 'em', 'rem', '%', '-', 'vw', 'vh', 'pt', 'pc' );
				//
				//              if ( ! $value ) {
				//                  return array(
				//                      'size' => '',
				//                      'unit' => 'px',
				//                  );
				//              }
				//
				//              preg_match( '/^(\d+(?:\.\d+)?)(.*)$/', $value, $matches );
				////
				//              $size = $matches[1];
				//              $unit = $matches[2];

				//              if ( 'px' !== $unit ) {
				//                  switch ( $unit ) {
				//                      case 'em':
				//                      case 'pc':
				//                      case 'rem':
				//                          $size *= 16;
				//                          $unit  = 'px';
				//                          break;
				//                      case 'vw':
				//                          $size *= 19.2;
				//                          $unit  = 'px';
				//                          break;
				//                      case 'vh':
				//                          $size *= 10.8;
				//                          $unit  = 'px';
				//                          break;
				//                      case '%':
				//                          $size *= 1.6;
				//                          $unit  = 'px';
				//                          break;
				//                      case 'pt':
				//                          $size *= 1.333;
				//                          $unit  = 'px';
				//                          break;
				//                      default:
				//                          // Do nothing if the unit is not recognized
				//                          break;
				//                  }
				//              }

				return array(
					'size' => $value,
					'unit' => 'px',
				);
			};

			$typography_options = array(
				array(
					'old_key' => 'colormag_widget_title_typography_setting',
					'new_key' => 'colormag_widget_title_typography',
				),
				array(
					'old_key' => 'colormag_base_typography_setting',
					'new_key' => 'colormag_base_typography',
				),
				array(
					'old_key' => 'colormag_headings_typography_setting',
					'new_key' => 'colormag_headings_typography',
				),
				array(
					'old_key' => 'colormag_h1_typography_setting',
					'new_key' => 'colormag_h1_typography',
				),
				array(
					'old_key' => 'colormag_h2_typography_setting',
					'new_key' => 'colormag_h2_typography',
				),
				array(
					'old_key' => 'colormag_h3_typography_setting',
					'new_key' => 'colormag_h3_typography',
				),
				array(
					'old_key' => 'colormag_h4_typography_setting',
					'new_key' => 'colormag_h4_typography',
				),
				array(
					'old_key' => 'colormag_h5_typography_setting',
					'new_key' => 'colormag_h5_typography',
				),
				array(
					'old_key' => 'colormag_h6_typography_setting',
					'new_key' => 'colormag_h6_typography',
				),
				array(
					'old_key' => 'colormag_button_typography_setting',
					'new_key' => 'colormag_button_typography',
				),
				array(
					'old_key' => 'colormag_site_title_typography_setting',
					'new_key' => 'colormag_site_title_typography',
				),
				array(
					'old_key' => 'colormag_site_tagline_typography_setting',
					'new_key' => 'colormag_site_tagline_typography',
				),
				array(
					'old_key' => 'colormag_breaking_news_label_typography_setting',
					'new_key' => 'colormag_news_ticker_label_typography',
				),
				array(
					'old_key' => 'colormag_breaking_news_content_typography_setting',
					'new_key' => 'colormag_news_ticker_content_typography',
				),
				array(
					'old_key' => 'colormag_primary_menu_typography_setting',
					'new_key' => 'colormag_primary_menu_typography',
				),
				array(
					'old_key' => 'colormag_primary_sub_menu_typography_setting',
					'new_key' => 'colormag_primary_sub_menu_typography',
				),
				array(
					'old_key' => 'colormag_post_title_typography_setting',
					'new_key' => 'colormag_post_title_typography',
				),
				array(
					'old_key' => 'colormag_comment_title_typography_setting',
					'new_key' => 'colormag_comment_title_typography',
				),
				array(
					'old_key' => 'colormag_post_meta_typography_setting',
					'new_key' => 'colormag_post_meta_typography',
				),
				array(
					'old_key' => 'colormag_page_title_typography_setting',
					'new_key' => 'colormag_page_title_typography',
				),
				array(
					'old_key' => 'colormag_footer_widget_title_typography_setting',
					'new_key' => 'colormag_footer_widget_title_typography',
				),
				array(
					'old_key' => 'colormag_footer_widget_content_typography_setting',
					'new_key' => 'colormag_footer_widget_content_typography',
				),
				array(
					'old_key' => 'colormag_footer_copyright_typography_setting',
					'new_key' => 'colormag_footer_copyright_typography',
				),
				array(
					'old_key' => 'colormag_footer_menu_typography_setting',
					'new_key' => 'colormag_footer_menu_typography',
				),
			);

			foreach ( $typography_options as $option ) {
				$old_value = get_theme_mod( $option['old_key'] );

				if ( $old_value ) {
					$new_desktop_font = isset( $old_value['font-size']['desktop'] ) ? $extractsizeandunit( $old_value['font-size']['desktop'] ) : array(
						'size' => '',
						'unit' => 'px',
					);

					$new_tablet_font = isset( $old_value['font-size']['tablet'] ) ? $extractsizeandunit( $old_value['font-size']['tablet'] ) : array(
						'size' => '',
						'unit' => 'px',
					);

					$new_mobile_font = isset( $old_value['font-size']['mobile'] ) ? $extractsizeandunit( $old_value['font-size']['mobile'] ) : array(
						'size' => '',
						'unit' => 'px',
					);

					$new_desktop_line_height = isset( $old_value['line-height']['desktop'] ) ? $extractsizeandunit( $old_value['line-height']['desktop'] ) : array(
						'size' => '',
						'unit' => '-',
					);

					if ( empty( $new_desktop_line_height['unit'] ) ) {
						$new_desktop_line_height['unit'] = '-';
					}

					$new_tablet_line_height = isset( $old_value['line-height']['tablet'] ) ? $extractsizeandunit( $old_value['line-height']['tablet'] ) : array(
						'size' => '',
						'unit' => '-',
					);

					if ( empty( $new_tablet_line_height['unit'] ) ) {
						$new_tablet_line_height['unit'] = '-';
					}

					$new_mobile_line_height = isset( $old_value['line-height']['mobile'] ) ? $extractsizeandunit( $old_value['line-height']['mobile'] ) : array(
						'size' => '',
						'unit' => '-',
					);

					if ( empty( $new_mobile_line_height['unit'] ) ) {
						$new_mobile_line_height['unit'] = '-';
					}

					$new_desktop_letter_spacing = isset( $old_value['letter-spacing']['desktop'] ) ? $extractsizeandunit( $old_value['letter-spacing']['desktop'] ) : array(
						'size' => '',
						'unit' => 'px',
					);

					$new_tablet_letter_spacing = isset( $old_value['letter-spacing']['tablet'] ) ? $extractsizeandunit( $old_value['letter-spacing']['tablet'] ) : array(
						'size' => '',
						'unit' => 'px',
					);

					$new_mobile_letter_spacing = isset( $old_value['letter-spacing']['mobile'] ) ? $extractsizeandunit( $old_value['letter-spacing']['mobile'] ) : array(
						'size' => '',
						'unit' => 'px',
					);

					$new_value = array(
						'font-family'    => isset( $old_value['font-family'] ) ? $old_value['font-family'] : 'default',
						'font-weight'    => isset( $old_value['font-weight'] ) ? $old_value['font-weight'] : 'default',
						'subsets'        => isset( $old_value['subsets'] ) ? $old_value['subsets'] : '',
						'font-size'      => array(
							'desktop' => array(
								'size' => $new_desktop_font['size'],
								'unit' => $new_desktop_font['unit'],
							),
							'tablet'  => array(
								'size' => $new_tablet_font['size'],
								'unit' => $new_tablet_font['unit'],
							),
							'mobile'  => array(
								'size' => $new_mobile_font['size'],
								'unit' => $new_mobile_font['unit'],
							),
						),
						'line-height'    => array(
							'desktop' => array(
								'size' => $new_desktop_line_height['size'],
								'unit' => '-',
							),
							'tablet'  => array(
								'size' => $new_tablet_line_height['size'],
								'unit' => '-',
							),
							'mobile'  => array(
								'size' => $new_mobile_line_height['size'],
								'unit' => '-',
							),
						),
						'letter-spacing' => array(
							'desktop' => array(
								'size' => $new_desktop_letter_spacing['size'],
								'unit' => $new_desktop_letter_spacing['unit'],
							),
							'tablet'  => array(
								'size' => $new_tablet_letter_spacing['size'],
								'unit' => $new_tablet_letter_spacing['unit'],
							),
							'mobile'  => array(
								'size' => $new_mobile_letter_spacing['size'],
								'unit' => $new_mobile_letter_spacing['unit'],
							),
						),
						'font-style'     => isset( $old_value['font-style'] ) ? $old_value['font-style'] : '',
						'text-transform' => isset( $old_value['text-transform'] ) ? $old_value['text-transform'] : '',
					);

					set_theme_mod( $option['new_key'], $new_value );
					remove_theme_mod( $option['old_key'] );
				}
			}

			// Dimension control migration.
			$dimension_option = array(
				array(
					'old_key' => 'colormag_primary_menu_dimension_padding',
					'new_key' => 'colormag_primary_menu_padding',
				),
			);

			foreach ( $dimension_option as $option ) {

				// Check if id exist in database or not.
				if ( ! array_key_exists( $option['old_key'], $this->old_theme_mods ) ) {
					continue;
				}

				$old_value = get_theme_mod( $option['old_key'] );

				// Check if the old value have unit key on or not.
				if ( false !== strpos( wp_json_encode( $old_value ), 'unit' ) ) {
					continue;
				}

				if ( $old_value ) {

					$new_top = isset( $old_value['top'] ) ? $extractsizeandunit( $old_value['top'] ) : array(
						'size' => '',
						'unit' => 'px',
					);

					$new_right  = isset( $old_value['right'] ) ? $extractsizeandunit( $old_value['right'] ) : array(
						'size' => '',
						'unit' => 'px',
					);
					$new_bottom = isset( $old_value['bottom'] ) ? $extractsizeandunit( $old_value['bottom'] ) : array(
						'size' => '',
						'unit' => 'px',
					);
					$new_left   = isset( $old_value['left'] ) ? $extractsizeandunit( $old_value['left'] ) : array(
						'size' => '',
						'unit' => 'px',
					);

					$new_value = array(
						'top'    => $new_top['size'],
						'right'  => $new_right['size'],
						'bottom' => $new_bottom['size'],
						'left'   => $new_left['size'],
						'unit'   => $new_top['unit'],
					);

					set_theme_mod( $option['new_key'], $new_value );

					if ( $option['old_key'] !== $option['new_key'] ) {

						remove_theme_mod( $option['old_key'] );
					}
				}
			}

			// Slider control migration.
			$slider_options = array(
				array(
					'old_key' => 'colormag_container_width',
					'new_key' => 'colormag_container_width',
					'default' => array(
						'size' => 1160,
						'unit' => 'px',
					),
				),

				array(
					'old_key' => 'colormag_site_logo_height',
					'new_key' => 'colormag_site_logo_height',
					'default' => array(
						'size' => 1160,
						'unit' => 'px',
					),
				),

				array(
					'old_key' => 'colormag_primary_menu_logo_height',
					'new_key' => 'colormag_primary_menu_logo_height',
					'default' => array(
						'size' => '',
						'unit' => 'px',
					),
				),

				array(
					'old_key' => 'colormag_primary_menu_logo_spacing',
					'new_key' => 'colormag_primary_menu_logo_spacing',
					'default' => array(
						'size' => '',
						'unit' => 'px',
					),
				),

				array(
					'old_key' => 'colormag_top_bar_border_bottom_size',
					'new_key' => 'colormag_top_bar_border_bottom_size',
					'default' => array(
						'size' => '',
						'unit' => 'px',
					),
				),
			);

			// Loop through the options and migrate their values.
			foreach ( $slider_options as $option ) {

				// Check if id exist in database or not.
				if ( ! array_key_exists( $option['old_key'], $this->old_theme_mods ) ) {
					continue;
				}

				$old_value = get_theme_mod( $option['old_key'] );

				// Check if the value is scalar.
				if ( ! is_scalar( $old_value ) ) {
					continue;
				}

				if ( $old_value ) {

					set_theme_mod(
						$option['new_key'],
						array(
							'size' => $old_value,
							'unit' => $option['default']['unit'],
						)
					);

					if ( $option['old_key'] !== $option['new_key'] ) {

						remove_theme_mod( $option['old_key'] );
					}
				}
			}

			/**
			 * Radio image control migration.
			 */
			// Sidebar Layout Migration.
			$sidebar_layout_option = array(
				array(
					'old_key' => 'colormag_default_layout',
					'new_key' => 'colormag_default_sidebar_layout',
				),
				array(
					'old_key' => 'colormag_default_page_layout',
					'new_key' => 'colormag_page_sidebar_layout',
				),
				array(
					'old_key' => 'colormag_default_single_posts_layout',
					'new_key' => 'colormag_post_sidebar_layout',
				),
			);

			foreach ( $sidebar_layout_option as $option ) {

				$old_value = get_theme_mod( $option['old_key'] );

				if ( $old_value ) {
					if ( 'right_sidebar' === $old_value ) {
						$new_value = 'right_sidebar';
					} elseif ( 'left_sidebar' === $old_value ) {
						$new_value = 'left_sidebar';
					} elseif ( 'no_sidebar_full_width' === $old_value ) {
						$new_value = 'no_sidebar_full_width';
					} elseif ( 'no_sidebar_content_centered' === $old_value ) {
						$new_value = 'no_sidebar_content_centered';
					} elseif ( 'two_sidebars' === $old_value ) {
						$new_value = 'two_sidebars';
					}

					set_theme_mod( $option['new_key'], $new_value );
					remove_theme_mod( $option['old_key'] );
				}
			}

			$theme_support_background_color = get_theme_mod( 'background_color' );
			$theme_support_background_image = get_theme_mod( 'background_image' );

			set_theme_mod(
				'colormag_outside_container_background',
				array(
					'background-color' => $theme_support_background_color,
					'background-image' => $theme_support_background_image,

				)
			);

			/**
			 * Background control migration.
			 */
			// Background Migration.
			$background_option = array(
				array(
					'old_key' => 'colormag_header_background_setting',
					'new_key' => 'colormag_main_header_background',
				),
				array(
					'old_key' => 'colormag_footer_copyright_background_setting',
					'new_key' => 'colormag_footer_copyright_background',
				),
				array(
					'old_key' => 'colormag_footer_background_setting',
					'new_key' => 'colormag_footer_background',
				),
				array(
					'old_key' => 'colormag_primary_sub_menu_background_setting',
					'new_key' => 'colormag_primary_sub_menu_background',
				),
				array(
					'old_key' => 'colormag_primary_menu_background_setting',
					'new_key' => 'colormag_primary_menu_background',
				),
				array(
					'old_key' => 'colormag_breaking_news_label_background_color',
					'new_key' => 'colormag_news_ticker_label_background',
				),
			);

			foreach ( $background_option as $option ) {

				$old_value = get_theme_mod( $option['old_key'] );

				if ( $old_value ) {

					$new_value = array(
						'background-color'      => $old_value['background-color'],
						'background-image'      => $old_value['background-image'],
						'background-repeat'     => $old_value['background-repeat'],
						'background-position'   => $old_value['background-position'],
						'background-size'       => $old_value['background-size'],
						'background-attachment' => $old_value['background-attachment'],
					);

					set_theme_mod( $option['new_key'], $new_value );
					remove_theme_mod( $option['old_key'] );
				}
			}

			// Set flag not to repeat the migration process, run it only once.
			update_option( 'colormag_major_update_migration_v3', true );
		}

		/**
		 * Migrates outside background settings to a new theme mod.
		 *
		 * This function handles the migration of various background-related theme mods
		 * to a single, consolidated theme mod. It performs the following operations:
		 *
		 * 1. Checks if the migration has already been performed to avoid duplicate migrations.
		 * 2. Retrieves individual background-related theme mods (color, image, preset, position, size, repeat, attachment).
		 * 3. If any of these theme mods exist, it consolidates them into a single array.
		 * 4. Sets the new consolidated theme mod 'colormag_outside_container_background'.
		 * 5. Removes the old individual theme mods.
		 * 6. Updates an option to mark the migration as complete.
		 *
		 * This migration is necessary to update the theme's handling of background settings,
		 * moving from individual settings to a more flexible, consolidated approach.
		 *
		 * @return void
		 *
		 * @since 4.0.9
		 */
		public function colormag_outside_background_migration() {

			if ( get_option( 'colormag_outside_background_migration' ) ) {
				return;
			}

			$background_color      = get_theme_mod( 'background_color' );
			$background_image      = get_theme_mod( 'background_image' );
			$background_preset     = get_theme_mod( 'background_preset' );
			$background_position   = get_theme_mod( 'background_position' );
			$background_size       = get_theme_mod( 'background_size' );
			$background_repeat     = get_theme_mod( 'background_repeat' );
			$background_attachment = get_theme_mod( 'background_attachment' );

			if ( $background_color || $background_image || $background_preset || $background_position || $background_size || $background_repeat || $background_attachment ) {
				$background_value = array(
					'background-color'      => $background_color,
					'background-image'      => $background_image,
					'background-repeat'     => $background_repeat,
					'background-position'   => $background_position,
					'background-size'       => $background_size,
					'background-attachment' => $background_attachment,
				);

				set_theme_mod( 'colormag_outside_container_background', $background_value );
				remove_theme_mod( 'background_color' );
				remove_theme_mod( 'background_image' );
				remove_theme_mod( 'background_preset' );
				remove_theme_mod( 'background_position' );
				remove_theme_mod( 'background_size' );
				remove_theme_mod( 'background_attachment' );
				remove_theme_mod( 'background_repeat' );
			}

			update_option( 'colormag_outside_background_migration', true );
		}

		/**
		 * Performs migration for the eLearning theme builder configuration.
		 *
		 * This function handles the migration of the header builder configuration
		 * from an older version to the current version. It performs the following tasks:
		 *
		 * 1. Checks if the migration has already been performed to avoid duplicate migrations.
		 * 2. Sets up a default header builder configuration for desktop and mobile layouts.
		 * 3. Migrates top bar settings if enabled, including left and right content.
		 * 4. Migrates main header settings based on the header style (left, center, right).
		 * 5. Handles the migration of primary menu, header search, and header button placements.
		 * 6. Ensures mobile menu configuration is set up properly.
		 *
		 * The function modifies theme mods to update the header builder configuration
		 * and removes old theme mods that are no longer needed after the migration.
		 *
		 * @return void
		 *
		 * @since 5.0.0
		 */
		public function colormag_builder_migration() {

			if ( get_option( 'colormag_builder_migration' ) && ! doing_action( 'themegrill_ajax_demo_imported' ) ) {
				return;
			}

			if ( ( doing_action( 'themegrill_ajax_demo_imported' ) && get_theme_mod( 'demo_migrated_to_builder' ) ) || get_option( 'colormag_free_demo_migrated_to_builder' ) ) {
				update_option( 'colormag_free_demo_migrated_to_builder', true );
				return;
			}

			$header_builder_config = [
				'desktop' => array(
					'top'    => array(
						'left'   => array(),
						'center' => array(),
						'right'  => array(),
					),
					'main'   => array(
						'left'   => array(
							'logo',
						),
						'center' => array(),
						'right'  => array(),
					),
					'bottom' => array(
						'left'   => array( 'primary-menu' ),
						'center' => array(),
						'right'  => array(),
					),
				),
				'mobile'  => array(
					'top'    => array(
						'left'   => array(),
						'center' => array(),
						'right'  => array(),
					),
					'main'   => array(
						'left'   => array(),
						'center' => array(),
						'right'  => array(),
					),
					'bottom' => array(
						'left'   => array(
							'toggle-button',
						),
						'center' => array(),
						'right'  => array(),
					),
				),
				'offset'  => array(
					'mobile-menu',
				),
			];

			$main_header_layout                   = get_theme_mod( 'colormag_main_header_layout', 'layout-1' );
			$home_icon                            = get_theme_mod( 'colormag_menu_icon_logo', 'icon' );
			$search_enable                        = get_theme_mod( 'colormag_enable_search', 0 );
			$random_enable                        = get_theme_mod( 'colormag_enable_random_post', 0 );
			$main_header_layout_style_1           = get_theme_mod( 'colormag_main_header_layout_1_style', 'style-1' );
			$main_header_layout_style_2           = get_theme_mod( 'colormag_main_header_layout_2_style', 'style-1' );
			$main_header_layout_1_style_alignment = get_theme_mod( 'colormag_header_display_type', 'type_one' );

			if ( 'layout-1' === $main_header_layout ) {
				if ( 'style-1' === $main_header_layout_style_1 ) {
					if ( 'type_one' === $main_header_layout_1_style_alignment ) {
						$bottom_left = [];
						self::remove_component( 'logo', $header_builder_config );
						$header_builder_config['desktop']['main']['left'][] = 'logo';

						if ( is_active_sidebar( 'colormag_header_sidebar' ) ) {
							self::remove_component( 'widget-1', $header_builder_config );
							$header_builder_config['desktop']['main']['right'][] = 'widget-1';
							$header_builder_config['mobile']['main']['center'][] = 'widget-1';
						}

						if ( 'home-icon' === $home_icon ) {
							self::remove_component( 'home-icon', $header_builder_config );
							$bottom_left[] = 'home-icon';
						}
						self::remove_component( 'primary-menu', $header_builder_config );
						$bottom_left[]                                      = 'primary-menu';
						$header_builder_config['desktop']['bottom']['left'] = $bottom_left;

						if ( $search_enable ) {
							self::remove_component( 'search', $header_builder_config );
							$header_builder_config['desktop']['bottom']['right'][] = 'search';
						}

						if ( $random_enable ) {
							self::remove_component( 'random', $header_builder_config );
							$header_builder_config['desktop']['bottom']['right'][] = 'random';
						}
					} elseif ( 'type_three' === $main_header_layout_1_style_alignment ) {
						$bottom_left = [];
						self::remove_component( 'logo', $header_builder_config );
						$header_builder_config['desktop']['main']['center'][] = 'logo';

						if ( is_active_sidebar( 'colormag_header_sidebar' ) ) {
							self::remove_component( 'widget-1', $header_builder_config );
							$header_builder_config['desktop']['main']['center'][] = 'widget-1';
							$header_builder_config['mobile']['main']['center'][]  = 'widget-1';
						}

						if ( 'home-icon' === $home_icon ) {
							self::remove_component( 'primary-menu', $header_builder_config );
							$bottom_left[] = 'home-icon';
						}
						$bottom_left[]                                      = 'primary-menu';
						$header_builder_config['desktop']['bottom']['left'] = $bottom_left;

						if ( $search_enable ) {
							self::remove_component( 'search', $header_builder_config );
							$header_builder_config['desktop']['bottom']['right'][] = 'search';
						}

						if ( $random_enable ) {
							self::remove_component( 'random', $header_builder_config );
							$header_builder_config['desktop']['bottom']['right'][] = 'random';
						}
					} elseif ( 'type_two' === $main_header_layout_1_style_alignment ) {
						$bottom_left = [];
						self::remove_component( 'logo', $header_builder_config );
						$header_builder_config['desktop']['main']['right'][] = 'logo';

						if ( is_active_sidebar( 'colormag_header_sidebar' ) ) {
							self::remove_component( 'widget-1', $header_builder_config );
							$header_builder_config['desktop']['main']['left'][]  = 'widget-1';
							$header_builder_config['mobile']['main']['center'][] = 'widget-1';
						}

						if ( 'home-icon' === $home_icon ) {
							self::remove_component( 'home-icon', $header_builder_config );
							$bottom_left[] = 'home-icon';
						}
						self::remove_component( 'primary-menu', $header_builder_config );
						$bottom_left[]                                      = 'primary-menu';
						$header_builder_config['desktop']['bottom']['left'] = $bottom_left;

						if ( $search_enable ) {
							self::remove_component( 'search', $header_builder_config );
							$header_builder_config['desktop']['bottom']['right'][] = 'search';
						}

						if ( $random_enable ) {
							self::remove_component( 'random', $header_builder_config );
							$header_builder_config['desktop']['bottom']['right'][] = 'random';
						}
					}
				} elseif ( 'style-2' === $main_header_layout_style_1 ) {
					if ( 'type_one' === $main_header_layout_1_style_alignment ) {
						$main_left = [];
						if ( 'home-icon' === $home_icon ) {
							self::remove_component( 'home-icon', $header_builder_config );
							$main_left[] = 'home-icon';
						}
						self::remove_component( 'primary-menu', $header_builder_config );
						$main_left[]                                      = 'primary-menu';
						$header_builder_config['desktop']['main']['left'] = $main_left;

						if ( $search_enable ) {
							self::remove_component( 'search', $header_builder_config );
							$header_builder_config['desktop']['main']['right'][] = 'search';
						}

						if ( $random_enable ) {
							self::remove_component( 'random', $header_builder_config );
							$header_builder_config['desktop']['main']['right'][] = 'random';
						}

						self::remove_component( 'logo', $header_builder_config );
						$header_builder_config['desktop']['bottom']['left'] = [ 'logo' ];

						if ( is_active_sidebar( 'colormag_header_sidebar' ) ) {
							self::remove_component( 'widget-1', $header_builder_config );
							$header_builder_config['desktop']['bottom']['right'][] = 'widget-1';
							$header_builder_config['mobile']['main']['center'][]   = 'widget-1';
						}
					} elseif ( 'type_three' === $main_header_layout_1_style_alignment ) {
						$main_left = [];
						if ( 'home-icon' === $home_icon ) {
							self::remove_component( 'home-icon', $header_builder_config );
							$main_left[] = 'home-icon';
						}
						self::remove_component( 'primary-menu', $header_builder_config );
						$main_left[]                                        = 'primary-menu';
						$header_builder_config['desktop']['main']['left'][] = $main_left;

						if ( $search_enable ) {
							self::remove_component( 'search', $header_builder_config );
							$header_builder_config['desktop']['main']['right'][] = 'search';
						}

						if ( $random_enable ) {
							self::remove_component( 'random', $header_builder_config );
							$header_builder_config['desktop']['main']['right'][] = 'random';
						}

						self::remove_component( 'logo', $header_builder_config );
						$header_builder_config['desktop']['bottom']['center'][] = 'logo';

						if ( is_active_sidebar( 'colormag_header_sidebar' ) ) {
							self::remove_component( 'widget-1', $header_builder_config );
							$header_builder_config['desktop']['bottom']['center'][] = 'widget-1';
							$header_builder_config['mobile']['main']['center'][]    = 'widget-1';
						}
					} elseif ( 'type_two' === $main_header_layout_1_style_alignment ) {
						$main_left = [];
						if ( 'home-icon' === $home_icon ) {
							self::remove_component( 'home-icon', $header_builder_config );
							$main_left[] = 'home-icon';
						}
						self::remove_component( 'primary-menu', $header_builder_config );
						$main_left[]                                      = 'primary-menu';
						$header_builder_config['desktop']['main']['left'] = $main_left;

						if ( $search_enable ) {
							self::remove_component( 'search', $header_builder_config );
							$header_builder_config['desktop']['main']['right'][] = 'search';
						}

						if ( $random_enable ) {
							self::remove_component( 'random', $header_builder_config );
							$header_builder_config['desktop']['main']['right'][] = 'random';
						}

						self::remove_component( 'logo', $header_builder_config );
						$header_builder_config['desktop']['bottom']['right'] = [ 'logo' ];

						if ( is_active_sidebar( 'colormag_header_sidebar' ) ) {
							self::remove_component( 'widget-1', $header_builder_config );
							$header_builder_config['desktop']['bottom']['left']  = [ 'widget-1' ];
							$header_builder_config['mobile']['main']['center'][] = 'widget-1';
						}
					}
				}
			} elseif ( 'layout-2' === $main_header_layout ) {
				if ( 'style-1' === $main_header_layout_style_2 ) {
					self::remove_component( 'logo', $header_builder_config );
					$header_builder_config['desktop']['main']['center'][] = 'logo';

					if ( is_active_sidebar( 'colormag_header_sidebar' ) ) {
						self::remove_component( 'widget-1', $header_builder_config );
						$header_builder_config['desktop']['main']['center'][] = 'widget-1';
						$header_builder_config['mobile']['main']['center'][]  = 'widget-1';
					}
					if ( 'home-icon' === $home_icon ) {
						self::remove_component( 'home-icon', $header_builder_config );
						$header_builder_config['desktop']['bottom']['left'] = [ 'home-icon' ];
					}
					self::remove_component( 'primary-menu', $header_builder_config );
					$header_builder_config['desktop']['bottom']['center'] = [ 'primary-menu' ];

					if ( $search_enable ) {
						self::remove_component( 'search', $header_builder_config );
						$header_builder_config['desktop']['bottom']['right'][] = 'search';
					}

					if ( $random_enable ) {
						self::remove_component( 'random', $header_builder_config );
						$header_builder_config['desktop']['bottom']['right'][] = 'random';
					}
				} elseif ( 'style-2' === $main_header_layout_style_2 ) {
					if ( 'home-icon' === $home_icon ) {
						self::remove_component( 'home-icon', $header_builder_config );
						$header_builder_config['desktop']['main']['left'] = [ 'home-icon' ];
					}
					self::remove_component( 'primary-menu', $header_builder_config );
					$header_builder_config['desktop']['main']['center'] = [ 'primary-menu' ];

					if ( $search_enable ) {
						self::remove_component( 'search', $header_builder_config );
						$header_builder_config['desktop']['main']['right'][] = 'search';
					}

					if ( $random_enable ) {
						self::remove_component( 'random', $header_builder_config );
						$header_builder_config['desktop']['main']['right'][] = 'random';
					}

					self::remove_component( 'logo', $header_builder_config );
					$header_builder_config['desktop']['bottom']['center'][] = 'logo';

					if ( is_active_sidebar( 'colormag_header_sidebar' ) ) {
						self::remove_component( 'widget-1', $header_builder_config );
						$header_builder_config['desktop']['bottom']['center'][] = 'widget-1';
						$header_builder_config['mobile']['main']['center'][]    = 'widget-1';
					}
				}
			} elseif ( 'layout-3' === $main_header_layout ) {
				$main_header_layout_style_3 = get_theme_mod( 'colormag_main_header_layout_3_style', 'style-1' );
				if ( 'style-1' === $main_header_layout_style_3 ) {

					if ( 'type_one' === $main_header_layout_1_style_alignment ) {
						$top_left = [];
						if ( 'home-icon' === $home_icon ) {
							self::remove_component( 'home-icon', $header_builder_config );
							$top_left[] = 'home-icon';
						}
						self::remove_component( 'primary-menu', $header_builder_config );
						$top_left[]                                      = 'primary-menu';
						$header_builder_config['desktop']['top']['left'] = $top_left;

						if ( $search_enable ) {
							self::remove_component( 'search', $header_builder_config );
							$header_builder_config['desktop']['top']['right'][] = 'search';
						}

						if ( $random_enable ) {
							self::remove_component( 'random', $header_builder_config );
							$header_builder_config['desktop']['top']['right'][] = 'random';
						}

						self::remove_component( 'logo', $header_builder_config );
						$header_builder_config['desktop']['bottom']['left'] = [ 'logo' ];

						if ( is_active_sidebar( 'colormag_header_sidebar' ) ) {
							self::remove_component( 'widget-1', $header_builder_config );
							$header_builder_config['desktop']['bottom']['right'][] = 'widget-1';
							$header_builder_config['mobile']['main']['center'][]   = 'widget-1';
						}
					} elseif ( 'type_three' === $main_header_layout_1_style_alignment ) {
						$top_left = [];
						if ( 'home-icon' === $home_icon ) {
							self::remove_component( 'home-icon', $header_builder_config );
							$top_left[] = 'home-icon';
						}
						self::remove_component( 'primary-menu', $header_builder_config );
						$top_left[]                                      = 'primary-menu';
						$header_builder_config['desktop']['top']['left'] = $top_left;

						if ( $search_enable ) {
							self::remove_component( 'search', $header_builder_config );
							$header_builder_config['desktop']['top']['right'][] = 'search';
						}

						if ( $random_enable ) {
							self::remove_component( 'random', $header_builder_config );
							$header_builder_config['desktop']['top']['right'][] = 'random';
						}

						self::remove_component( 'logo', $header_builder_config );
						$header_builder_config['desktop']['bottom']['center'][] = 'logo';

						if ( is_active_sidebar( 'colormag_header_sidebar' ) ) {
							self::remove_component( 'widget-1', $header_builder_config );
							$header_builder_config['desktop']['bottom']['center'][] = 'widget-1';
							$header_builder_config['mobile']['main']['center'][]    = 'widget-1';
						}
					} elseif ( 'type_two' === $main_header_layout_1_style_alignment ) {
						$top_left = [];
						if ( 'home-icon' === $home_icon ) {
							self::remove_component( 'home-icon', $header_builder_config );
							$top_left[] = 'home-icon';
						}
						self::remove_component( 'primary-menu', $header_builder_config );
						$top_left[]                                      = 'primary-menu';
						$header_builder_config['desktop']['top']['left'] = $top_left;

						if ( $search_enable ) {
							self::remove_component( 'search', $header_builder_config );
							$header_builder_config['desktop']['top']['right'][] = 'search';
						}

						if ( $random_enable ) {
							self::remove_component( 'random', $header_builder_config );
							$header_builder_config['desktop']['top']['right'][] = 'random';
						}

						self::remove_component( 'logo', $header_builder_config );
						$header_builder_config['desktop']['bottom']['right'] = [ 'logo' ];

						if ( is_active_sidebar( 'colormag_header_sidebar' ) ) {
							self::remove_component( 'widget-1', $header_builder_config );
							$header_builder_config['desktop']['bottom']['left'][] = 'widget-1';
							$header_builder_config['mobile']['main']['center'][]  = 'widget-1';
						}
					}
				}
			}

			$header_builder_config['mobile']['main']['center'][] = 'logo';
			if ( count( $header_builder_config['mobile']['main']['center'] ) > 1 ) {
				$mobile_builder_first                                 = $header_builder_config['mobile']['main']['center'][0];
				$header_builder_config['mobile']['main']['center'][0] = $header_builder_config['mobile']['main']['center'][1];
				$header_builder_config['mobile']['main']['center'][1] = $mobile_builder_first;
			}

			$top_bar_enable     = get_theme_mod( 'colormag_enable_top_bar', 0 );
			$date_enable        = get_theme_mod( 'colormag_date_display', false );
			$news_ticker_enable = get_theme_mod( 'colormag_enable_news_ticker', 0 );
			$social_enable      = get_theme_mod( 'colormag_enable_social_icons', 0 );
			$top_bar_menu       = get_theme_mod( 'colormag_top_bar_menu_enable', 0 );
			if ( $top_bar_enable ) {

				if ( 'layout-3' === $main_header_layout ) {
					if ( $date_enable ) {
						self::remove_component( 'date', $header_builder_config );
						$header_builder_config['desktop']['main']['left'][] = 'date';
					}

					if ( $news_ticker_enable ) {
						self::remove_component( 'news-ticker', $header_builder_config );
						$header_builder_config['desktop']['main']['left'][] = 'news-ticker';
					}

					if ( $top_bar_menu ) {
						self::remove_component( 'secondary-menu', $header_builder_config );
						$header_builder_config['desktop']['main']['right'][] = 'secondary-menu';
					}

					if ( $social_enable ) {
						self::remove_component( 'socials', $header_builder_config );
						$header_builder_config['desktop']['main']['right'][] = 'socials';
					}
				} else {
					if ( $date_enable ) {
						self::remove_component( 'date', $header_builder_config );
						$header_builder_config['desktop']['top']['left'][] = 'date';
					}

					if ( $news_ticker_enable ) {
						self::remove_component( 'news-ticker', $header_builder_config );
						$header_builder_config['desktop']['top']['left'][] = 'news-ticker';
					}

					if ( $top_bar_menu ) {
						self::remove_component( 'secondary-menu', $header_builder_config );
						$header_builder_config['desktop']['top']['right'][] = 'secondary-menu';
					}

					if ( $social_enable ) {
						self::remove_component( 'socials', $header_builder_config );
						$header_builder_config['desktop']['top']['right'][] = 'socials';
					}
				}
			}

			if ( $top_bar_menu ) {
				$menu_locations = get_theme_mod( 'nav_menu_locations' );

				// Check if 'menu-mobile' is set
				if ( ! isset( $menu_locations['menu-secondary'] ) && isset( $menu_locations['top-bar'] ) ) {
					// Set 'menu-secondary' to the value of 'primary' menu location
					$menu_locations['menu-secondary'] = $menu_locations['top-bar'];

					// Update the theme mod with the new menu locations
					set_theme_mod( 'nav_menu_locations', $menu_locations );
				}
			}

			// Get the current menu locations
			$menu_locations = get_theme_mod( 'nav_menu_locations' );

			// Check if 'menu-mobile' is set
			if ( ! isset( $menu_locations['menu-mobile'] ) && isset( $menu_locations['primary'] ) ) {
				// Set 'menu-mobile' to the value of 'primary' menu location
				$menu_locations['menu-mobile'] = $menu_locations['primary'];

				// Update the theme mod with the new menu locations
				set_theme_mod( 'nav_menu_locations', $menu_locations );
			}

			set_theme_mod( 'colormag_header_builder', $header_builder_config );

			// Footer builder migration.
			$footer_builder_config = [
				'desktop' => [
					'top'    => [
						'top-1' => [],
						'top-2' => [],
						'top-3' => [],
						'top-4' => [],
						'top-5' => [],
					],
					'main'   => [
						'main-1' => [],
						'main-2' => [],
						'main-3' => [],
						'main-4' => [],
						'main-5' => [],
					],
					'bottom' => [
						'bottom-1' => [],
						'bottom-2' => [],
						'bottom-3' => [],
						'bottom-4' => [],
						'bottom-5' => [],
					],
				],
				'offset'  => [],
			];

			set_theme_mod( 'colormag_footer_top_area_cols', 3 );
			if ( is_active_sidebar( 'colormag_footer_sidebar_one_upper' ) ) {
				$footer_builder_config['desktop']['top']['top-1'][] = 'widget-1';
			}

			if ( is_active_sidebar( 'colormag_footer_sidebar_two_upper' ) ) {
				$footer_builder_config['desktop']['top']['top-2'][] = 'widget-2';
			}

			if ( is_active_sidebar( 'colormag_footer_sidebar_three_upper' ) ) {
				$footer_builder_config['desktop']['top']['top-3'][] = 'widget-3';
			}

			$footer_column_layout = get_theme_mod( 'colormag_footer_column_layout', 'style-4' );
			if ( 'style-1' === $footer_column_layout ) {
				set_theme_mod( 'colormag_footer_main_area_cols', 1 );
				if ( is_active_sidebar( 'colormag_footer_sidebar_one' ) ) {
					$footer_builder_config['desktop']['main']['main-1'][] = 'widget-4';
				}
			} elseif ( 'style-2' === $footer_column_layout ) {
				set_theme_mod( 'colormag_footer_main_area_cols', 2 );
				if ( is_active_sidebar( 'colormag_footer_sidebar_one' ) ) {
					$footer_builder_config['desktop']['main']['main-1'][] = 'widget-4';
				}
				if ( is_active_sidebar( 'colormag_footer_sidebar_two' ) ) {
					$footer_builder_config['desktop']['main']['main-2'][] = 'widget-5';
				}
			} elseif ( 'style-3' === $footer_column_layout ) {
				set_theme_mod( 'colormag_footer_main_area_cols', 3 );
				if ( is_active_sidebar( 'colormag_footer_sidebar_one' ) ) {
					$footer_builder_config['desktop']['main']['main-1'][] = 'widget-4';
				}
				if ( is_active_sidebar( 'colormag_footer_sidebar_two' ) ) {
					$footer_builder_config['desktop']['main']['main-2'][] = 'widget-5';
				}
				if ( is_active_sidebar( 'colormag_footer_sidebar_three' ) ) {
					$footer_builder_config['desktop']['main']['main-3'][] = 'widget-6';
				}
			} elseif ( 'style-4' === $footer_column_layout ) {
				set_theme_mod( 'colormag_footer_main_area_cols', 4 );
				set_theme_mod( 'colormag_footer_main_column_layout_style_4', 'style-9' );
				if ( is_active_sidebar( 'colormag_footer_sidebar_one' ) ) {
					$footer_builder_config['desktop']['main']['main-1'][] = 'widget-4';
				}
				if ( is_active_sidebar( 'colormag_footer_sidebar_two' ) ) {
					$footer_builder_config['desktop']['main']['main-2'][] = 'widget-5';
				}
				if ( is_active_sidebar( 'colormag_footer_sidebar_three' ) ) {
					$footer_builder_config['desktop']['main']['main-3'][] = 'widget-6';
				}
				if ( is_active_sidebar( 'colormag_footer_sidebar_four' ) ) {
					$footer_builder_config['desktop']['main']['main-4'][] = 'widget-7';
				}
			}

			$footer_bar_alignment       = get_theme_mod( 'colormag_footer_bar_alignment', 'left' );
			$social_icons_enable        = get_theme_mod( 'colormag_enable_social_icons', false );
			$social_icons_footer_enable = get_theme_mod( 'colormag_enable_social_icons_footer', true );
			$menu_locations             = get_theme_mod( 'nav_menu_locations' );
			if ( 'left' === $footer_bar_alignment ) {
				$footer_builder_config['desktop']['bottom']['bottom-1'] = [ 'copyright' ];
				if ( $social_icons_enable ) {
					if ( $social_icons_footer_enable ) {
						$footer_builder_config['desktop']['bottom']['bottom-2'][] = 'socials';
					}
				}
				if ( isset( $menu_locations['footer'] ) ) {
					self::remove_component( 'footer-menu', $footer_builder_config );
					$footer_builder_config['desktop']['bottom']['bottom-2'][] = 'footer-menu';
					$menu_locations['footer-menu']                            = $menu_locations['footer'];

					// Update the theme mod with the new menu locations
					set_theme_mod( 'nav_menu_locations', $menu_locations );
				}
			} elseif ( 'right' === $footer_column_layout ) {
				if ( $social_icons_enable ) {
					if ( $social_icons_footer_enable ) {
						$footer_builder_config['desktop']['bottom']['bottom-1'][] = 'socials';
					}
				}
				if ( isset( $menu_locations['footer'] ) ) {
					self::remove_component( 'footer-menu', $footer_builder_config );
					$footer_builder_config['desktop']['bottom']['bottom-1'][] = 'footer-menu';
					$menu_locations['footer-menu']                            = $menu_locations['footer'];

					// Update the theme mod with the new menu locations
					set_theme_mod( 'nav_menu_locations', $menu_locations );
				}
				$footer_builder_config['desktop']['bottom']['bottom-2'] = [ 'copyright' ];

			} elseif ( 'center' === $footer_column_layout ) {
				set_theme_mod( 'colormag_footer_bottom_area_cols', 1 );
				if ( isset( $menu_locations['footer'] ) ) {
					self::remove_component( 'footer-menu', $footer_builder_config );
					$footer_builder_config['desktop']['bottom']['bottom-1'][] = 'footer-menu';
					$menu_locations['footer-menu']                            = $menu_locations['footer'];

					// Update the theme mod with the new menu locations
					set_theme_mod( 'nav_menu_locations', $menu_locations );
				}
				$footer_builder_config['desktop']['bottom']['bottom-1'][] = 'copyright';
				if ( $social_icons_enable ) {
					if ( $social_icons_footer_enable ) {
						$footer_builder_config['desktop']['bottom']['bottom-1'][] = 'socials';
					}
				}
			}

			// Social links lists.
			$social_links_count    = 70;
			$colormag_social_links = array(
				'colormag_social_facebook'    => array(
					'id'      => 'colormag_social_facebook',
					'title'   => esc_html__( 'Facebook', 'colormag' ),
					'default' => '',
				),
				'colormag_social_twitter'     => array(
					'id'      => 'colormag_social_twitter',
					'title'   => esc_html__( 'Twitter', 'colormag' ),
					'default' => '',
				),
				'colormag_social_instagram'   => array(
					'id'      => 'colormag_social_instagram',
					'title'   => esc_html__( 'Instagram', 'colormag' ),
					'default' => '',
				),
				'colormag_social_pinterest'   => array(
					'id'      => 'colormag_social_pinterest',
					'title'   => esc_html__( 'Pinterest', 'colormag' ),
					'default' => '',
				),
				'colormag_social_youtube'     => array(
					'id'      => 'colormag_social_youtube',
					'title'   => esc_html__( 'YouTube', 'colormag' ),
					'default' => '',
				),
				'colormag_social_vimeo'       => array(
					'id'      => 'colormag_social_vimeo',
					'title'   => esc_html__( 'Vimeo-Square', 'colormag' ),
					'default' => '',
				),
				'colormag_social_linkedin'    => array(
					'id'      => 'colormag_social_linkedin',
					'title'   => esc_html__( 'LinkedIn', 'colormag' ),
					'default' => '',
				),
				'colormag_social_delicious'   => array(
					'id'      => 'colormag_social_delicious',
					'title'   => esc_html__( 'Delicious', 'colormag' ),
					'default' => '',
				),
				'colormag_social_flickr'      => array(
					'id'      => 'colormag_social_flickr',
					'title'   => esc_html__( 'Flickr', 'colormag' ),
					'default' => '',
				),
				'colormag_social_skype'       => array(
					'id'      => 'colormag_social_skype',
					'title'   => esc_html__( 'Skype', 'colormag' ),
					'default' => '',
				),
				'colormag_social_soundcloud'  => array(
					'id'      => 'colormag_social_soundcloud',
					'title'   => esc_html__( 'SoundCloud', 'colormag' ),
					'default' => '',
				),
				'colormag_social_vine'        => array(
					'id'      => 'colormag_social_vine',
					'title'   => esc_html__( 'Vine', 'colormag' ),
					'default' => '',
				),
				'colormag_social_stumbleupon' => array(
					'id'      => 'colormag_social_stumbleupon',
					'title'   => esc_html__( 'StumbleUpon', 'colormag' ),
					'default' => '',
				),
				'colormag_social_tumblr'      => array(
					'id'      => 'colormag_social_tumblr',
					'title'   => esc_html__( 'Tumblr', 'colormag' ),
					'default' => '',
				),
				'colormag_social_reddit'      => array(
					'id'      => 'colormag_social_reddit',
					'title'   => esc_html__( 'Reddit', 'colormag' ),
					'default' => '',
				),
				'colormag_social_xing'        => array(
					'id'      => 'colormag_social_xing',
					'title'   => esc_html__( 'Xing', 'colormag' ),
					'default' => '',
				),
				'colormag_social_vk'          => array(
					'id'      => 'colormag_social_vk',
					'title'   => esc_html__( 'VK', 'colormag' ),
					'default' => '',
				),
				'colormag_social_discord'     => array(
					'id'      => 'colormag_social_discord',
					'title'   => esc_html__( 'Discord', 'colormag' ),
					'default' => '',
				),
			);

			$all_social_links = [];
			foreach ( $colormag_social_links as $colormag_social_link ) {
				$have_url = get_theme_mod( $colormag_social_link['id'] );

				if ( $have_url ) {
					// Extract the social media name from the ID
					$social_name = str_replace( 'colormag_social_', '', $colormag_social_link['id'] );

					$icon = $social_name;

					if ( 'twitter' === $social_name ) {
						$icon = 'x-twitter';
					} elseif ( 'instagram' === $social_name ) {
						$icon = 'square-instagram';
					}

					$footer_social_value = array(
						'id'    => uniqid(),
						'label' => $social_name,
						'url'   => $have_url,
						'icon'  => 'fa-brands fa-' . $icon,
					);

					// You might want to add this value to a larger array of social links
					$all_social_links[] = $footer_social_value;
				}
			}

			$colormag_additional_social_link = array(
				'user_custom_social_links_one'   => __( 'Additional Social Link One', 'colormag' ),
				'user_custom_social_links_two'   => __( 'Additional Social Link Two', 'colormag' ),
				'user_custom_social_links_three' => __( 'Additional Social Link Three', 'colormag' ),
				'user_custom_social_links_four'  => __( 'Additional Social Link Four', 'colormag' ),
				'user_custom_social_links_five'  => __( 'Additional Social Link Five', 'colormag' ),
				'user_custom_social_links_six'   => __( 'Additional Social Link Six', 'colormag' ),
			);
			foreach ( $colormag_additional_social_link as $key => $value ) {
				$link      = get_theme_mod( $key, '' );
				$icon_name = get_theme_mod( $key . '_fontawesome' );
				if ( ! empty( $link ) ) {
					$footer_social_value = array(
						'id'    => uniqid(),
						'label' => $icon_name,
						'url'   => $link,
						'icon'  => 'fa-brands fa-' . $icon_name,
					);
					// You might want to add this value to a larger array of social links
					$all_social_links[] = $footer_social_value;
				}
			}

			set_theme_mod( 'colormag_header_socials', $all_social_links );
			set_theme_mod( 'colormag_footer_socials', $all_social_links );

			set_theme_mod( 'colormag_footer_builder', $footer_builder_config );

			// Option migration.
			$top_bar_background = get_theme_mod( 'colormag_top_bar_background_color', '' );
			if ( $top_bar_background ) {
				$top_bar_background_value = array(
					'background-color'      => $top_bar_background,
					'background-image'      => '',
					'background-repeat'     => 'repeat',
					'background-position'   => 'center center',
					'background-size'       => 'contain',
					'background-attachment' => 'scroll',
				);

				set_theme_mod( 'colormag_header_top_area_background', $top_bar_background_value );
				remove_theme_mod( 'colormag_top_bar_background_color' );
			}

			$site_title_color = get_theme_mod( 'colormag_site_title_color', '' );
			if ( $site_title_color ) {
				set_theme_mod( 'colormag_header_site_identity_color', $site_title_color );
				remove_theme_mod( 'colormag_site_title_color' );
			}

			$site_title_hover_color = get_theme_mod( 'colormag_site_title_hover_color', '' );
			if ( $site_title_hover_color ) {
				set_theme_mod( 'colormag_header_site_identity_hover_color', $site_title_hover_color );
				remove_theme_mod( 'colormag_site_title_hover_color' );
			}

			$site_title_typography = get_theme_mod( 'colormag_site_title_typography', '' );
			if ( $site_title_typography ) {
				set_theme_mod( 'colormag_header_site_title_typography', $site_title_typography );
				remove_theme_mod( 'colormag_site_title_typography' );
			}

			$site_tagline_color = get_theme_mod( 'colormag_site_tagline_color', '' );
			if ( $site_tagline_color ) {
				set_theme_mod( 'colormag_header_site_tagline_color', $site_tagline_color );
				remove_theme_mod( 'colormag_site_tagline_color' );
			}

			$site_tagline_typography = get_theme_mod( 'colormag_site_tagline_typography', '' );
			if ( $site_tagline_typography ) {
				set_theme_mod( 'colormag_header_site_tagline_typography', $site_tagline_typography );
				remove_theme_mod( 'colormag_site_tagline_typography' );
			}

			$main_header_background = get_theme_mod( 'colormag_main_header_background', '' );
			if ( $main_header_background ) {
				set_theme_mod( 'colormag_header_main_area_background', $main_header_background );
				remove_theme_mod( 'colormag_main_header_background' );
			}

			$main_header_width = get_theme_mod( 'colormag_main_header_width_setting', 'full-width' );
			if ( $main_header_width ) {
				set_theme_mod( 'colormag_main_header_width_setting', $main_header_width );
				remove_theme_mod( 'colormag_main_header_width_setting' );
			}

			$primary_menu_bg = get_theme_mod(
				'colormag_primary_menu_background',
				array(
					'background-color'      => '#27272A',
					'background-image'      => '',
					'background-position'   => 'center center',
					'background-size'       => 'auto',
					'background-attachment' => 'scroll',
					'background-repeat'     => 'repeat',
				)
			);
			if ( $primary_menu_bg ) {
				if ( 'layout-3' === $main_header_layout ) {
					set_theme_mod( 'colormag_header_top_area_background', $primary_menu_bg );
				} else {
					set_theme_mod( 'colormag_header_bottom_area_background', $primary_menu_bg );
				}
				remove_theme_mod( 'colormag_primary_menu_background' );
			}

			$primary_menu_border = get_theme_mod(
				'colormag_primary_menu_top_border_width',
				array(
					'size'  => '4',
					'units' => 'px',
				)
			);
			if ( $primary_menu_border ) {
				$value = array(
					'top'    => $primary_menu_border['size'],
					'right'  => '0',
					'bottom' => '0',
					'left'   => '0',
					'unit'   => 'px',
				);
				if ( 'layout-3' === $main_header_layout ) {
					set_theme_mod( 'colormag_header_top_area_border_width', $value );
				} else {
					set_theme_mod( 'colormag_header_bottom_area_border_width', $value );
				}
				remove_theme_mod( 'colormag_primary_menu_top_border_width' );
			}

			$primary_menu_border_color = get_theme_mod( 'colormag_primary_menu_top_border_color' );
			$primary_color             = get_theme_mod( 'colormag_primary_color' );
			if ( 'layout-3' === $main_header_layout ) {
				if ( $primary_menu_border_color ) {
					set_theme_mod( 'colormag_header_top_area_border_color', $primary_menu_border_color );
					remove_theme_mod( 'colormag_primary_menu_top_border_color' );
				} else {
					set_theme_mod( 'colormag_header_top_area_border_color', $primary_color );
				}
			} elseif ( $primary_menu_border_color ) {
				set_theme_mod( 'colormag_header_bottom_area_border_color', $primary_menu_border_color );
				remove_theme_mod( 'colormag_primary_menu_top_border_color' );
			} else {
				set_theme_mod( 'colormag_header_bottom_area_border_color', $primary_color );
			}

			$primary_menu_color = get_theme_mod( 'colormag_primary_menu_text_color' );
			if ( $primary_menu_color ) {
				set_theme_mod( 'colormag_header_primary_menu_text_color', $primary_menu_color );
				remove_theme_mod( 'colormag_primary_menu_text_color' );
			}

			$primary_menu_hover_color = get_theme_mod( 'colormag_primary_menu_selected_hovered_text_color' );
			if ( $primary_menu_hover_color ) {
				set_theme_mod( 'colormag_header_primary_menu_selected_hovered_text_color', $primary_menu_hover_color );
				remove_theme_mod( 'colormag_primary_menu_selected_hovered_text_color' );
			}

			$primary_menu_typo = get_theme_mod( 'colormag_primary_menu_typography' );
			if ( $primary_menu_typo ) {
				set_theme_mod( 'colormag_header_primary_menu_typography', $primary_menu_typo );
				remove_theme_mod( 'colormag_primary_menu_typography' );
			}

			$primary_sub_menu_bg = get_theme_mod(
				'colormag_primary_sub_menu_background',
				array(
					'background-color'      => '#232323',
					'background-image'      => '',
					'background-position'   => 'center center',
					'background-size'       => 'auto',
					'background-attachment' => 'scroll',
					'background-repeat'     => 'repeat',
				)
			);

			if ( $primary_sub_menu_bg ) {
				set_theme_mod( 'colormag_header_primary_sub_menu_background', $primary_sub_menu_bg );
				remove_theme_mod( 'colormag_primary_sub_menu_background' );
			}

			$primary_sub_menu_typography = get_theme_mod( 'colormag_primary_sub_menu_typography' );
			if ( $primary_sub_menu_typography ) {
				set_theme_mod( 'colormag_primary_sub_menu_typography', $primary_sub_menu_typography );
				remove_theme_mod( 'colormag_primary_sub_menu_typography' );
			}

			$header_toggle_color = get_theme_mod( 'colormag_mobile_menu_toggle_icon_color', '#fff' );
			if ( $header_toggle_color ) {
				set_theme_mod( 'colormag_header_builder_toggle_button_color', $header_toggle_color );
				remove_theme_mod( 'colormag_mobile_menu_toggle_icon_color' );
			}

			$mobile_menu_text_color = get_theme_mod( 'colormag_mobile_menu_text_color' );
			if ( $mobile_menu_text_color ) {
				set_theme_mod( 'colormag_header_mobile_menu_item_color', $mobile_menu_text_color );
				remove_theme_mod( 'colormag_mobile_menu_text_color' );
			}

			$mobile_menu_text_hover_color = get_theme_mod( 'colormag_mobile_menu_selected_hovered_text_color' );
			if ( $mobile_menu_text_hover_color ) {
				set_theme_mod( 'colormag_header_mobile_menu_item_hover_color', $mobile_menu_text_hover_color );
				remove_theme_mod( 'colormag_mobile_menu_selected_hovered_text_color' );
			}

			$mobile_typography = get_theme_mod( 'colormag_mobile_menu_typography' );
			if ( $mobile_typography ) {
				set_theme_mod( 'colormag_header_mobile_menu_typography', $mobile_typography );
				remove_theme_mod( 'colormag_mobile_menu_typography' );
			}

			$header_action_color = get_theme_mod( 'colormag_header_action_icon_color', '#fff' );
			if ( $header_action_color ) {
				set_theme_mod( 'colormag_header_search_icon_color', $header_action_color );
				set_theme_mod( 'colormag_header_random_icon_color', $header_action_color );
				remove_theme_mod( 'colormag_header_action_icon_color' );
			}

			$header_action_hover_color = get_theme_mod( 'colormag_header_action_icon_hover_color', '' );
			if ( $header_action_hover_color ) {
				set_theme_mod( 'colormag_header_search_icon_hover_color', $header_action_hover_color );
				set_theme_mod( 'colormag_header_random_icon_hover_color', $header_action_hover_color );
				remove_theme_mod( 'colormag_header_action_icon_hover_color' );
			}

			$footer_background = get_theme_mod( 'colormag_footer_background', '' );

			if ( $footer_background ) {
				set_theme_mod( 'colormag_footer_top_area_background', $footer_background );
				set_theme_mod( 'colormag_footer_main_area_background', $footer_background );
				remove_theme_mod( 'colormag_footer_background' );
			}

			$upper_footer_background = get_theme_mod( 'colormag_upper_footer_background', '' );
			if ( $upper_footer_background ) {
				set_theme_mod( 'colormag_footer_top_area_widget_background', $upper_footer_background );
				remove_theme_mod( 'colormag_upper_footer_background' );
			}

			$upper_footer_color = get_theme_mod( 'colormag_upper_footer_color', '#ffffff' );
			if ( $upper_footer_color ) {
				set_theme_mod( 'colormag_footer_top_area_color', $upper_footer_color );
				remove_theme_mod( 'colormag_upper_footer_color' );
			}

			$widget_title_color = get_theme_mod( 'colormag_footer_widget_title_color', '' );
			if ( $widget_title_color ) {
				set_theme_mod( 'colormag_footer_widget_1_title_color', $widget_title_color );
				set_theme_mod( 'colormag_footer_widget_2_title_color', $widget_title_color );
				set_theme_mod( 'colormag_footer_widget_3_title_color', $widget_title_color );
				set_theme_mod( 'colormag_footer_widget_4_title_color', $widget_title_color );
				set_theme_mod( 'colormag_footer_widget_5_title_color', $widget_title_color );
				set_theme_mod( 'colormag_footer_widget_6_title_color', $widget_title_color );
				set_theme_mod( 'colormag_footer_widget_7_title_color', $widget_title_color );
				remove_theme_mod( 'colormag_footer_widget_title_color' );
			}

			$widget_title_typography = get_theme_mod( 'colormag_footer_widget_title_typography', '' );
			if ( $widget_title_typography ) {
				set_theme_mod( 'colormag_footer_widget_1_title_typography', $widget_title_typography );
				set_theme_mod( 'colormag_footer_widget_2_title_typography', $widget_title_typography );
				set_theme_mod( 'colormag_footer_widget_3_title_typography', $widget_title_typography );
				set_theme_mod( 'colormag_footer_widget_4_title_typography', $widget_title_typography );
				set_theme_mod( 'colormag_footer_widget_5_title_typography', $widget_title_typography );
				set_theme_mod( 'colormag_footer_widget_6_title_typography', $widget_title_typography );
				set_theme_mod( 'colormag_footer_widget_7_title_typography', $widget_title_typography );
				remove_theme_mod( 'colormag_footer_widget_title_typography' );
			}

			$widget_content_color = get_theme_mod( 'colormag_footer_widget_content_color', '' );
			if ( $widget_content_color ) {
				set_theme_mod( 'colormag_footer_widget_1_content_color', $widget_content_color );
				set_theme_mod( 'colormag_footer_widget_2_content_color', $widget_content_color );
				set_theme_mod( 'colormag_footer_widget_3_content_color', $widget_content_color );
				set_theme_mod( 'colormag_footer_widget_4_content_color', $widget_content_color );
				set_theme_mod( 'colormag_footer_widget_5_content_color', $widget_content_color );
				set_theme_mod( 'colormag_footer_widget_6_content_color', $widget_content_color );
				set_theme_mod( 'colormag_footer_widget_7_content_color', $widget_content_color );
				remove_theme_mod( 'colormag_footer_widget_content_color' );
			}

			$widget_content_typography = get_theme_mod( 'colormag_footer_widget_1_content_typography', '' );
			if ( $widget_content_typography ) {
				set_theme_mod( 'colormag_footer_widget_1_content_typography', $widget_content_typography );
				set_theme_mod( 'colormag_footer_widget_2_content_typography', $widget_content_typography );
				set_theme_mod( 'colormag_footer_widget_3_content_typography', $widget_content_typography );
				set_theme_mod( 'colormag_footer_widget_4_content_typography', $widget_content_typography );
				set_theme_mod( 'colormag_footer_widget_5_content_typography', $widget_content_typography );
				set_theme_mod( 'colormag_footer_widget_6_content_typography', $widget_content_typography );
				set_theme_mod( 'colormag_footer_widget_7_content_typography', $widget_content_typography );
				remove_theme_mod( 'colormag_footer_widget_1_content_typography' );
			}

			$widget_link_color = get_theme_mod( 'colormag_footer_widget_content_link_text_color', '' );
			if ( $widget_link_color ) {
				set_theme_mod( 'colormag_footer_widget_1_link_color', $widget_link_color );
				set_theme_mod( 'colormag_footer_widget_2_link_color', $widget_link_color );
				set_theme_mod( 'colormag_footer_widget_3_link_color', $widget_link_color );
				set_theme_mod( 'colormag_footer_widget_4_link_color', $widget_link_color );
				set_theme_mod( 'colormag_footer_widget_5_link_color', $widget_link_color );
				set_theme_mod( 'colormag_footer_widget_6_link_color', $widget_link_color );
				set_theme_mod( 'colormag_footer_widget_7_link_color', $widget_link_color );
				remove_theme_mod( 'colormag_footer_widget_content_link_text_color' );
			}

			$widget_link_hover_color = get_theme_mod( 'colormag_footer_widget_content_link_text_hover_color', '' );
			if ( $widget_link_hover_color ) {
				set_theme_mod( 'colormag_footer_widget_1_link_hover_color', $widget_link_hover_color );
				set_theme_mod( 'colormag_footer_widget_2_link_hover_color', $widget_link_hover_color );
				set_theme_mod( 'colormag_footer_widget_3_link_hover_color', $widget_link_hover_color );
				set_theme_mod( 'colormag_footer_widget_4_link_hover_color', $widget_link_hover_color );
				set_theme_mod( 'colormag_footer_widget_5_link_hover_color', $widget_link_hover_color );
				set_theme_mod( 'colormag_footer_widget_6_link_hover_color', $widget_link_hover_color );
				set_theme_mod( 'colormag_footer_widget_7_link_hover_color', $widget_link_hover_color );
				remove_theme_mod( 'colormag_footer_widget_content_link_text_hover_color' );
			}

			$footer_bar_bg_color = get_theme_mod( 'colormag_footer_copyright_background', '' );
			if ( $footer_bar_bg_color ) {
				set_theme_mod( 'colormag_footer_bottom_area_background', $footer_bar_bg_color );
				remove_theme_mod( 'colormag_footer_copyright_background' );
			}

			$footer_bar_link_color = get_theme_mod( 'colormag_footer_copyright_link_text_color', '' );
			if ( $footer_bar_link_color ) {
				set_theme_mod( 'colormag_footer_copyright_link_color', $footer_bar_link_color );
				remove_theme_mod( 'colormag_footer_copyright_link_text_color' );
			}

			update_option( 'colormag_builder_migration', true );
		}

		/**
		 * Recursively removes a specified component from an array.
		 *
		 * This static function traverses through a multidimensional array and removes
		 * all occurrences of a specified component. It performs the following operations:
		 *
		 * 1. Iterates through each element of the input array.
		 * 2. If an element is an array, it recursively calls itself on that sub-array.
		 * 3. If an element matches the component to remove, it unsets that element.
		 * 4. After processing, if the array keys are sequential integers, it reindexes the array.
		 *
		 * @param mixed $component_to_remove The component to be removed from the array.
		 * @param array &$_array             The array to remove the component from (passed by reference).
		 *
		 * @return void The function modifies the input array directly.
		 *
		 * @since 5.0.0
		 */
		public static function remove_component( $component_to_remove, &$_array ) {
			foreach ( $_array as $key => &$value ) {
				if ( is_array( $value ) ) {
					self::remove_component( $component_to_remove, $value );
				} else { // phpcs:ignore
					if ( $value === $component_to_remove ) {
						unset( $_array[ $key ] );
					}
				}
			}
			if ( array_values( $_array ) === $_array ) {
				$_array = array_values( $_array );
			}
		}

		/**
		 * Return the value for customize migration on demo import.
		 *
		 * @return bool
		 */
		function colormag_demo_import_migration() {
			if ( isset( $_GET['demo-import-migration'] ) && isset( $_GET['_demo_import_migration_nonce'] ) ) {
				if ( ! wp_verify_nonce( $_GET['_demo_import_migration_nonce'], 'demo_import_migration' ) ) {
					wp_die( __( 'Action failed. Please refresh the page and retry.', 'colormag' ) );
				}

				return true;
			}

			return false;
		}

		/**
		 * Determines whether to run the customizer migration.
		 *
		 * This static function checks if the customizer migration needs to be executed.
		 * It performs the following checks:
		 *
		 * 1. Verifies if the migration has already been run by checking a specific option.
		 * 2. If the migration has been run before, it returns false.
		 * 3. If not previously migrated, it checks for the presence of old theme mods.
		 * 4. Specifically looks for theme mods with the 'colormag_' prefix.
		 *
		 * The function is designed to prevent unnecessary migrations and ensure
		 * that the migration only runs when old theme data is present.
		 *
		 * @return bool Returns true if migration should be run, false otherwise.
		 *
		 */
		public static function maybe_run_migration() {

			/**
			 * Check migration is already run or not.
			 * If migration is already run then return false.
			 *
			 */
			$migrated = get_option( 'colormag_major_update_migration_v3' ) || get_theme_mod( 'colormag_enable_builder' );

			if ( $migrated || wp_doing_ajax() ) {

				return false;
			}

			/**
			 * If user don't import the demo and upgrade the theme.
			 * Then we need to run the migration.
			 *
			 */
			$result     = false;
			$theme_mods = get_theme_mods();

			foreach ( $theme_mods as $key => $_ ) {

				if ( strpos( $key, 'colormag_' ) !== false ) {

					$result = true;
					break;
				}
			}

			return $result;
		}
	}

	new ColorMag_Migration();
}
